/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

package org.opensi.facturation.commerciaux;

import java.awt.Color;
import java.io.IOException;
import java.sql.Date;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;

import org.opensi.util.pdf.MultiPagesDocument;
import org.opensi.util.pdf.PdfWidths;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.Chunk;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;

public class CommissionStd extends MultiPagesDocument {

	private static final Font fontNomEntreprise = new Font(Font.HELVETICA, 10, Font.BOLD);
	private static final Font fontNomCommercial = new Font(Font.HELVETICA, 10, Font.BOLD);
	private static final Font fontInfosCommercial = new Font(Font.HELVETICA, 10, Font.NORMAL);
	private static final Font fontInfosEntreprise = new Font(Font.HELVETICA, 10, Font.NORMAL);
	private static final Font fontIntitule = new Font(Font.HELVETICA, 14, Font.BOLD, Color.WHITE);
	private static final Font fontNomColonne = new Font(Font.HELVETICA, 10, Font.BOLD);
	private static final Font fontLine = new Font(Font.HELVETICA , 8, Font.NORMAL);
	private static final Font fontLineTotal = new Font(Font.HELVETICA , 8, Font.BOLD);
	private static final Font fontLegale = new Font(Font.HELVETICA, 8, Font.NORMAL);
	private static final Font fontTotaux = new Font(Font.HELVETICA , 10, Font.NORMAL);
	private static final Font fontCommission = new Font(Font.HELVETICA , 10, Font.BOLD);
	private static final Font fontWeb = new Font(Font.HELVETICA, 10, Font.ITALIC);
		
	private DecimalFormat numFormat = new DecimalFormat("0.00");
	private DecimalFormat numFormatCap = new DecimalFormat("0.##");
	private SimpleDateFormat formatDate = new SimpleDateFormat("dd/MM/yyyy");

	private DataCommission id;
	private Image logo;
	
	private float[] widthsCols;
	private int nbCols;

	public CommissionStd(String tempdir, DataCommission dc) throws DocumentException {

		super(tempdir);
		
		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');
		
		numFormat.setDecimalFormatSymbols(dfs);
		numFormat.setNegativeSuffix("");
		numFormat.setNegativePrefix("-");
		numFormat.setPositiveSuffix(" ");
		numFormat.setGroupingUsed(true);
		numFormat.setGroupingSize(3);
		
		numFormatCap.setDecimalFormatSymbols(dfs);
		numFormatCap.setGroupingUsed(true);
		numFormatCap.setGroupingSize(3);

		this.id = dc;
		
		try {
			logo = Image.getInstance(id.urlLogo);
			if (logo.getDpiX()!=0 && logo.getDpiY()!=0)
				logo.scaleAbsolute(logo.getPlainWidth() * 72 / logo.getDpiX(), logo.getPlainHeight() * 72 / logo.getDpiY());
		}
		catch (IOException e) {
			logo = null;
		}
		
		PdfWidths pw = new PdfWidths();
		
		pw.addColonneFixe(25);
		if (!id.filtre.equals("ND")) {
			pw.addColonneVariableMin(4,20);
			if (id.filtre.equals("F2")) {
				pw.addColonneVariableMin(4,20);
			} else if (id.filtre.equals("F3")) {
				pw.addColonneVariableMin(4,20);
				pw.addColonneVariableMin(4,20);
			}
		}
		pw.addColonneFixe(20);
		pw.addColonneVariableMin(1, 20);
		
		nbCols = pw.getNbCols();
		widthsCols = pw.getWidthsCols();
	}
	
	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(35));
		tb.add(makeTitre());
		tb.add(makeSpace(5));		
		tb.add(makeEntetesColonnes());		
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(firstHeader());
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
	
		tb.add(lastLine());
		tb.add(makeTotaux());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());
		
		return tb.getTable();
	}	
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(lastLine());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());		
		
		return tb.getTable();		
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
		
		LigneCommission lc = id.lignes.get(numLine);
		
		switch (lc.type) {			
		case 'P':	return makeLinePeriode(lc);
		case 'V': return makeLineVente(lc);
		case 'D':
		default: return makeLineDetail(lc);
		}
	}
	
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));		
	
		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}
	
		return pptSpace;
	}
	
	
	public int nbLines() {
		return id.getNbLignes();
	}	
	
	public int nbEx() {
		return 1;
	}

	public Image imageFond() {
		return null;
	}
	
	
	/* mthodes internes */

	private PdfPTable makeSpace(int space) {
	
		PdfPTable pptSpace = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));
	
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	
	public PdfPTable lastLine() throws DocumentException {

		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT+PdfPCell.BOTTOM);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(1);
		cell.setPhrase(new Phrase(""));		

		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}

		return pptSpace;
	}
	

	public PdfPTable makeEntete() throws DocumentException {
		
		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		
		PdfPCell cellCommercial = new PdfPCell(makeInfosCommercial());
		cellCommercial.setBorder(PdfPCell.NO_BORDER);
		
		pptEntete.addCell(cellSociete);		
		pptEntete.addCell(cellCommercial);
		
		return pptEntete;
	}	
	
	
	private PdfPTable makeInfosSociete() {
		
		PdfPTable pptInfos = new PdfPTable(1);
		
		PdfPCell cell;
		
		if (logo != null) {
			cell = new PdfPCell(logo, false);

			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

			pptInfos.addCell(cell);

			cell.setImage(null);

			cell.setMinimumHeight(5);
			cell.setPhrase(new Phrase("", fontInfosEntreprise));
			pptInfos.addCell(cell);
			cell.setMinimumHeight(0);
		}
		else {
			cell = new PdfPCell(pptInfos.getDefaultCell());
			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		}		
		
		if (!id.logoAdr) {
			cell.setPhrase(new Phrase(id.raisonSociale, fontNomEntreprise));		
			pptInfos.addCell(cell);
			cell.setPhrase(new Phrase(id.adresse1, fontInfosEntreprise));		
			pptInfos.addCell(cell);
			if (!id.adresse2.equals("")) {
				cell.setPhrase(new Phrase(id.adresse2, fontInfosEntreprise));
				pptInfos.addCell(cell);
			}
			if (!id.adresse3.equals("")) {
				cell.setPhrase(new Phrase(id.adresse3, fontInfosEntreprise));
				pptInfos.addCell(cell);
			}
			cell.setPhrase(new Phrase(id.codePostal+" "+id.ville, fontInfosEntreprise));		
			pptInfos.addCell(cell);

			String telfax = "";

			if (id.tel.length()>0) {
				telfax = "T\u00E9l: "+ id.tel;
				if (id.fax.length()>0) {
					telfax += " - Fax: "+ id.fax;
				}
			}
			else if (id.fax.length()>0) {
				telfax = "Fax: "+ id.fax;
			}		

			if (telfax.length()>0) {
				cell.setPhrase(new Phrase(telfax, fontInfosEntreprise));		
				pptInfos.addCell(cell);
			}

			if (!id.email.equals("")) {
				cell.setPhrase(new Phrase("e-mail: "+ id.email, fontInfosEntreprise));		
				pptInfos.addCell(cell);
			}
			if (!id.web.equals("")) {
				cell.setPhrase(new Phrase(id.web, fontWeb));		
				pptInfos.addCell(cell);
			}
		}
		
		cell.setPhrase(new Phrase(""));		
		pptInfos.addCell(cell);
		
		return pptInfos;	
	}
	

	private PdfPTable makeInfosCommercial() throws DocumentException {
		PdfPTable pptCommercial = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptCommercial.getDefaultCell());

		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase("Page " + currentPage + " / " + totalPages, fontInfosEntreprise));
		pptCommercial.addCell(cell);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(55);
		pptCommercial.addCell(cell);
		cell.setMinimumHeight(0);
		cell.setPaddingLeft(0);
		cell.setPaddingBottom(0);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.nomCommercial, fontNomCommercial));
		pptCommercial.addCell(cell);
		cell.setPhrase(new Phrase(id.adresse1Commercial, fontInfosCommercial));
		pptCommercial.addCell(cell);
		if (!id.adresse2Commercial.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Commercial, fontInfosCommercial));
			pptCommercial.addCell(cell);
		}
		cell.setPhrase(new Phrase(id.codePostalCommercial + " " + id.villeCommercial, fontInfosCommercial));
		pptCommercial.addCell(cell);
		cell.setPhrase(new Phrase(id.paysCommercial, fontInfosCommercial));
		pptCommercial.addCell(cell);
		
		cell.setPhrase(new Phrase("", fontLine));
		pptCommercial.addCell(cell);
		
		return pptCommercial;		
	}
	

	private PdfPTable makeTitre() throws DocumentException {
		
		PdfPTable pptTitre = new PdfPTable(1);
		pptTitre.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptTitre.getDefaultCell());		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setFixedHeight(30);
		cell.setPaddingBottom(5);
		cell.setPhrase(new Phrase("COMMISSIONNEMENT DU "+ formatDate.format(new Date(id.dateDebut)) +" AU "+ formatDate.format(new Date(id.dateFin)), fontIntitule));
		
		pptTitre.addCell(cell);
		
		return pptTitre;

	}

	private PdfPTable makeEntetesColonnes() throws DocumentException {

		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(this.widthsCols);
		
		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingBottom(4);
		cell.setMinimumHeight(40);
		
		cell.setPhrase(new Phrase("P\u00E9riode", fontNomColonne));
		pptCols.addCell(cell);
		
		if (id.filtre.equals("C")) {
			cell.setPhrase(new Phrase("Client", fontNomColonne));
			pptCols.addCell(cell);
		}
		else if (id.filtre.equals("MQ")) {
			cell.setPhrase(new Phrase("Marque", fontNomColonne));
			pptCols.addCell(cell);
		}
		else if (id.filtre.equals("F1")) {
			cell.setPhrase(new Phrase("Famille 1", fontNomColonne));
			pptCols.addCell(cell);
		}
		else if (id.filtre.equals("F2")) {
			cell.setPhrase(new Phrase("Famille 1", fontNomColonne));
			pptCols.addCell(cell);
			cell.setPhrase(new Phrase("Famille 2", fontNomColonne));
			pptCols.addCell(cell);
		}
		else if (id.filtre.equals("F3")) {
			cell.setPhrase(new Phrase("Famille 1", fontNomColonne));
			pptCols.addCell(cell);
			cell.setPhrase(new Phrase("Famille 2", fontNomColonne));
			pptCols.addCell(cell);
			cell.setPhrase(new Phrase("Famille 3", fontNomColonne));
			pptCols.addCell(cell);
		}
		else if (id.filtre.equals("A")) {
			cell.setPhrase(new Phrase("Article", fontNomColonne));
			pptCols.addCell(cell);
		}
		
		
		cell.setPhrase(new Phrase("N\u00B0 Facture", fontNomColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Commission", fontNomColonne));
		pptCols.addCell(cell);		
		
		return pptCols;
	}
	
	
	private PdfPTable makeLinePeriode(LigneCommission lc) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.BOX-PdfPCell.BOTTOM);
		cell.setBackgroundColor(new Color(153,153,153));

		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		
		cell.setPhrase(new Phrase(""+ formatDate.format(new Date(lc.dateDebut)) + " - " + formatDate.format(new Date(lc.dateFin)), fontLineTotal));
		pptLine.addCell(cell);

		if (!id.filtre.equals("ND")) {
			cell.setPhrase(new Phrase(""));
			pptLine.addCell(cell);
			
			if (id.filtre.equals("F2")) {
				pptLine.addCell(cell);
			} else if (id.filtre.equals("F3")) {
				pptLine.addCell(cell);
				pptLine.addCell(cell);
			}
		}
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		
		cell.setPaddingRight(4);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase(numFormat.format(lc.totalCommission), fontLineTotal));		
		pptLine.addCell(cell);

		return pptLine;	
	}
	
	
	private PdfPTable makeLineVente(LigneCommission lc) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setBackgroundColor(new Color(204,204,204));

		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);

		if (id.filtre.equals("C")) {
			cell.setPhrase(new Phrase(lc.refClient, fontLine));
			pptLine.addCell(cell);
		}
		else if (id.filtre.equals("MQ")) {
			if (lc.marque.equals("")) {
				cell.setPhrase(new Phrase(lc.enStock?"[SANS MARQUE]":"[HORS STOCK]", fontLine));
			} else {
				cell.setPhrase(new Phrase(lc.marque, fontLine));
			}
			pptLine.addCell(cell);
		}
		else if (id.filtre.equals("F1")) {
			// le cas "SANS FAMILLE 1" ne peut pas se produire
			cell.setPhrase(new Phrase(lc.famille1.equals("")?"[HORS STOCK]":lc.famille1, fontLine));
			pptLine.addCell(cell);
		}
		else if (id.filtre.equals("F2")) {
			// le cas "SANS FAMILLE 1" ne peut pas se produire
			cell.setPhrase(new Phrase(lc.famille1.equals("")?"[HORS STOCK]":lc.famille1, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lc.famille2.equals("") && lc.enStock?"[SANS FAMILLE 2]":lc.famille2, fontLine));
			pptLine.addCell(cell);
		}
		else if (id.filtre.equals("F3")) {
			// le cas "SANS FAMILLE 1" ne peut pas se produire
			cell.setPhrase(new Phrase(lc.famille1.equals("")?"[HORS STOCK]":lc.famille1, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lc.famille2.equals("") && lc.enStock?"[SANS FAMILLE 2]":lc.famille2, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lc.famille3.equals("") && lc.enStock?"[SANS FAMILLE 3]":lc.famille3, fontLine));
			pptLine.addCell(cell);
		}
		else if (id.filtre.equals("A")) {
			cell.setPhrase(new Phrase(lc.refArticle, fontLine));
			pptLine.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);

		cell.setPaddingRight(4);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		if (id.filtre.equals("C") && !lc.detailCommissionPossible) {
			cell.setPhrase(new Phrase("Calcul impossible", fontLine));
		} else {
			cell.setPhrase(new Phrase(numFormat.format(lc.totalCommission), fontLine));
		}
		pptLine.addCell(cell);

		return pptLine;	
	}
	
	
	private PdfPTable makeLineDetail(LigneCommission lc) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);

		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);

		if (!id.filtre.equals("ND")) {
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			
			if (id.filtre.equals("F2")) {
				pptLine.addCell(cell);
			} else if (id.filtre.equals("F3")) {
				pptLine.addCell(cell);
				pptLine.addCell(cell);
			}
		}
		
		cell.setPhrase(new Phrase(lc.refFacture, fontLine));
		pptLine.addCell(cell);

		cell.setPaddingRight(4);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		if (lc.detailCommissionPossible) {
			cell.setPhrase(new Phrase(numFormat.format(lc.totalCommission), fontLine));		
			pptLine.addCell(cell);
		} else {
			cell.setPhrase(new Phrase("Calcul impossible", fontLine));
			pptLine.addCell(cell);
		}

		return pptLine;	
	}
	
	
	private PdfPTable makeTotaux() throws DocumentException {
		
		PdfPTable pptTotaux = new PdfPTable(2);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {60,40};
		pptTotaux.setWidths(widths);		
			
		PdfPCell cell = new PdfPCell(makeZoneCommentaire());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);
		pptTotaux.addCell(cell);

		cell = new PdfPCell(makeCadreTotaux());
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);		
		pptTotaux.addCell(cell);		
				
		return pptTotaux;	
	}	
	
	
	private PdfPTable makeZoneCommentaire() throws DocumentException {		

		PdfPTable pptCom = new PdfPTable(1);
		pptCom.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptCom.getDefaultCell());	
		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(id.commentaires, fontLine));
		pptCom.addCell(cell);

		return pptCom;		
	}	


	private PdfPTable makeCadreTotaux() throws DocumentException {
		
		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		
		Chunk c1 = new Chunk("Total HT", fontTotaux);
		Chunk c2 = new Chunk (" (hors frais de port) ", fontLine);
		Chunk c3 = new Chunk (":", fontTotaux);
		Phrase p1 = new Phrase();
		p1.add(c1); p1.add(c2); p1.add(c3);
		cell.setPhrase(p1);
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(id.totalHT) +" \u20AC", fontTotaux));
		pptCadreTotaux.addCell(cell);

		Chunk c4 = new Chunk ("Total TTC", fontTotaux);
		Phrase p2 = new Phrase();
		p2.add(c4); p2.add(c2); p2.add(c3);
		cell.setPhrase(p2);
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(id.totalTTC) +" \u20AC", fontTotaux));
		pptCadreTotaux.addCell(cell);
		
		if (id.ajustement != 0) {
			cell.setPhrase(new Phrase("Commissionnement :", fontCommission));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.montantCommission) +" \u20AC", fontCommission));
			pptCadreTotaux.addCell(cell);
			
			cell.setPhrase(new Phrase("Ajustement :", fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.ajustement) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);
			
			cell.setPhrase(new Phrase("Total Commissions :", fontCommission));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.montantCommission + id.ajustement) +" \u20AC", fontCommission));
			pptCadreTotaux.addCell(cell);
		}
		else {
			cell.setPhrase(new Phrase("Total Commissions :", fontCommission));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.montantCommission) +" \u20AC", fontCommission));
			pptCadreTotaux.addCell(cell);
		}

		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);
		
		return pptCadreTotaux;
	}


	private PdfPTable makeInfosLegales() {
		
		PdfPTable pptLegales = new PdfPTable(1);
		pptLegales.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptLegales.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(id.raisonSociale +" - "+ id.typeSociete + (id.capital>0?" au capital de "+ numFormatCap.format(id.capital) +" Euros":"") +" - immatricul\u00E9e au RCS "+ id.rcs +" "+ id.siren +" - N\u00B0 TVA : "+ id.tvaIntra, fontLegale));
		
		pptLegales.addCell(cell);
		
		return pptLegales;	
	}

}
