/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration d'un bon de prparation au format standard
 */

package org.opensi.facturation.actions.documents.modeles;


import java.awt.Color;
import java.io.IOException;
import java.sql.Date;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;

import org.opensi.util.pdf.PdfWidths;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.Chunk;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class BonPreparationStd extends BonPreparationPDF {


private static final Font fontAdrFacture = new Font(Font.HELVETICA, 11, Font.BOLD);
private static final Font fontAdrSociete = new Font(Font.HELVETICA, 10, Font.NORMAL);
private static final Font fontAdrSocieteGras = new Font(Font.HELVETICA, 11, Font.BOLD);
private static final Font fontTitre = new Font(Font.HELVETICA , 16, Font.BOLD, Color.WHITE);
private static final Font fontWeb = new Font(Font.HELVETICA, 10, Font.ITALIC);
private static final Font fontCadre1 = new Font(Font.HELVETICA, 10, Font.BOLD);
private static final Font fontEnteteColonne = new Font(Font.HELVETICA, 9, Font.BOLD);
private static final Font fontInfosFact = new Font(Font.HELVETICA , 10, Font.NORMAL);
private static final Font fontLegale = new Font(Font.HELVETICA , 8, Font.NORMAL);
private static final Font fontLine = new Font(Font.HELVETICA , 8, Font.NORMAL);
private static final Font fontArtNom = new Font(Font.HELVETICA, 9, Font.ITALIC);
private static final Font fontSmallTitre = new Font(Font.HELVETICA, 12, Font.BOLD + Font.ITALIC, new Color(255, 255, 255));


private DecimalFormat numFormat = new DecimalFormat("0.00");
private DecimalFormat numFormatCap = new DecimalFormat("0.##");
private SimpleDateFormat formatDate = new SimpleDateFormat("dd/MM/yyyy");

private float[] widthsCols;
private int nbCols;

private boolean coul;
private boolean tail;
private boolean unite;
private boolean nbPieces;
private boolean numLot;
private boolean datePeremption;

private Image logo;



  public BonPreparationStd(String tempdir, DataBonPreparation id) throws Exception {

		super(tempdir, id);
		
		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');
		
		numFormat.setDecimalFormatSymbols(dfs);
		numFormat.setNegativeSuffix("");
		numFormat.setNegativePrefix("-");
		numFormat.setPositiveSuffix(" ");
		numFormat.setGroupingUsed(true);
		numFormat.setGroupingSize(3);
		
		numFormatCap.setDecimalFormatSymbols(dfs);
		numFormatCap.setGroupingUsed(true);
		numFormatCap.setGroupingSize(3);
		
		try {
			logo = Image.getInstance(id.urlLogo);
			if (logo.getDpiX()!=0 && logo.getDpiY()!=0)
      	logo.scaleAbsolute(logo.getPlainWidth() * 72 / logo.getDpiX(), logo.getPlainHeight() * 72 / logo.getDpiY());
		}
		catch (IOException e) {
			logo = null;
		}
		
		
		tail = id.tail;
		coul = id.coul;
		unite = id.unite;
		nbPieces = id.nbPieces;
		numLot = id.numLot;
		datePeremption = id.datePeremption;
		
		PdfWidths pw = new PdfWidths();
		
		pw.addColonneVariableMinMax(2, 13, 20);
		pw.addColonneVariableMinMax(2, 13, 20);
		if (id.referenceFabricant) pw.addColonneVariableMinMax(2, 13, 20);
		pw.addColonneVariableMin(4,20);		
		if (coul) pw.addColonneVariableMin(1, 7);
		if (tail) pw.addColonneVariableMin(1, 7);
		if (numLot) pw.addColonneVariableMinMax(2, 6, 9);
		if (nbPieces) pw.addColonneFixe(5);
		if (datePeremption) pw.addColonneFixe(8);
		pw.addColonneVariableMin(1, 8);
		if (unite) pw.addColonneFixe(5);
		if (id.stockReel) pw.addColonneVariableMinMax(1, 8, 12);
		if (id.stockDispo) pw.addColonneVariableMinMax(1, 8, 12);
		
		nbCols = pw.getNbCols();
		widthsCols = pw.getWidthsCols();
	}


	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		if (id.stockDispo || id.stockReel || id.referenceFabricant) {
			tb.add(makeEnteteWeb());
		} else {
			tb.add(makeEntete());
		}
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		tb.add(makeResponsable());
		tb.add(makeSpace(5));
		tb.add(makeEntetesColonnes());
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		if (id.stockDispo || id.stockReel || id.referenceFabricant) {
			tb.add(makeEnteteWeb());
		} else {
			tb.add(makeEntete());
		}
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		tb.add(makeEntetesColonnes());
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		if (id.comFin.length()>0 || id.mentions.length()>0) {
			tb.add(makeZoneCommentaire());			
		}
		
		tb.add(emptyLine(10));
		tb.add(makeFinTableau());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());
		
		return tb.getTable();
	}
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(emptyLine(10));
		tb.add(makeFinTableau());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());		
		
		return tb.getTable();		
	}	
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		LigneBonPreparation lbp = id.lignes.get(numLine);
	
		switch (lbp.type) {			
			case 'A':	return makeLineArticle(lbp);
			case 'N': return makeLineNomenclature(lbp);
			case 'C':
			default: return makeLineCommentaire(lbp);
		}		
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));		
	
		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}
	
		return pptSpace;
	}
	
	
	public int nbLines() {
	
		return id.getNbLignes();
	}
	
	
	public int nbEx() {
	
		return 1;
	}


	public Image imageFond() {
		if (id.annule) {	
			try {
				Image fond = Image.getInstance(id.urlAnnule);
				fond.scaleAbsolute(324, 468);
				fond.setAbsolutePosition(135,100);
				
				return fond;
			}
			catch (Exception e) {
				return null;
			}
		}
		else
			return null;
	}


	/* mthodes internes */

	private PdfPTable makeSpace(int space) {
	
		PdfPTable pptSpace = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));
	
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	

	private PdfPTable makeEnteteWeb() throws DocumentException {
	
		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		
		PdfPCell cellFacture = new PdfPCell(makeInfosFactureWeb());
		cellFacture.setBorder(PdfPCell.NO_BORDER);
		
		pptEntete.addCell(cellSociete);		
		pptEntete.addCell(cellFacture);			

		PdfPCell cellAdrFact = new PdfPCell(makeAdrFactWeb());
		cellAdrFact.setBorder(PdfPCell.NO_BORDER);

		PdfPCell cellAdrLiv = new PdfPCell(makeAdrLivWeb());
		cellAdrLiv.setBorder(PdfPCell.NO_BORDER);

		pptEntete.addCell(cellAdrFact);
		pptEntete.addCell(cellAdrLiv);
		
		PdfPCell cellVide = new PdfPCell(pptEntete.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		cellVide.setColspan(2);
		cellVide.setFixedHeight(10);
		pptEntete.addCell(cellVide);
		
		
		return pptEntete;		
	}
	
	
	private PdfPTable makeEntete() throws DocumentException {

		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		
		PdfPTable pptEnteteGauche = new PdfPTable(1);
		pptEnteteGauche.setWidthPercentage(100);
		PdfPTable pptEnteteDroite = new PdfPTable(1);
		pptEnteteDroite.setWidthPercentage(100);
		
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setMinimumHeight(115);
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		pptEnteteGauche.addCell(cellSociete);
		PdfPCell cellVide = new PdfPCell(pptEnteteGauche.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		pptEnteteGauche.addCell(cellVide);
		
		PdfPCell cellFacture = new PdfPCell(makeInfosFacture());
		cellFacture.setFixedHeight(115);
		cellFacture.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellFacture);		
		PdfPCell cellAdrEnvoi = new PdfPCell(makeAdrLiv());
		cellAdrEnvoi.setMinimumHeight(80);
		cellAdrEnvoi.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellAdrEnvoi);
		pptEnteteDroite.addCell(cellVide);
		
		PdfPCell cellPartieGauche = new PdfPCell(pptEnteteGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellPartieDroite = new PdfPCell(pptEnteteDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		pptEntete.addCell(cellPartieGauche);
		pptEntete.addCell(cellPartieDroite);
		
		return pptEntete;	
	}


	private PdfPTable makeInfosFactureWeb() throws DocumentException {

		PdfPTable pptInfosFacture = new PdfPTable(2);
		
		int[] widths = {60,40};
		pptInfosFacture.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingLeft(4);
		cell.setPaddingBottom(4);
		
		cell.setPhrase(new Phrase("BP du BL N\u00B0 "+ id.numeroBon, fontInfosFact));
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase("Date : "+ formatDate.format(new Date(id.dateBon)), fontInfosFact));
		pptInfosFacture.addCell(cell);
		
		cell.setColspan(2);
		if (id.clientId.length()>0) {
			cell.setPhrase(new Phrase("Client N\u00B0 "+ id.clientId +" - "+ id.raisonSocialeFact, fontInfosFact));
		}
		else {
			cell.setPhrase(new Phrase("Client : "+ id.raisonSocialeFact, fontInfosFact));
		}
		pptInfosFacture.addCell(cell);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(5);
		pptInfosFacture.addCell(cell);
		cell.setMinimumHeight(0);
		
		cell.setPhrase(new Phrase("Page "+currentPage+" / "+totalPages, fontInfosFact));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		pptInfosFacture.addCell(cell);


		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(65);
		pptInfosFacture.addCell(cell);
		cell.setMinimumHeight(0);

		cell.setPhrase(new Phrase("", fontLine));
		pptInfosFacture.addCell(cell);

		return pptInfosFacture;
	}


	private PdfPTable makeAdrFactWeb() throws DocumentException {

		PdfPTable pptInfosFacture = new PdfPTable(1);

		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setBackgroundColor(new Color(175, 175, 175));

		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPhrase(new Phrase("FACTURATION", fontSmallTitre));
		pptInfosFacture.addCell(cell);

		cell.setBackgroundColor(new Color(255, 255, 255));

		cell.setPaddingBottom(0);
		cell.setPaddingLeft(15);

		cell.setBorder(PdfPCell.BOX - PdfPCell.BOTTOM);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.raisonSocialeFact + " - " + id.interlocuteurFact, fontAdrFacture));
		pptInfosFacture.addCell(cell);

		cell.setBorder(PdfPCell.LEFT + PdfPCell.RIGHT);

		cell.setPhrase(new Phrase(id.adresseFact, fontAdrFacture));
		pptInfosFacture.addCell(cell);

		if (!id.compAdresseFact.equals("")) {
			cell.setPhrase(new Phrase(id.compAdresseFact, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}

		if (!id.adresseFact3.equals("")) {
			cell.setPhrase(new Phrase(id.adresseFact3, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}

		cell.setPhrase(new Phrase(id.codePostalFact +" "+ id.villeFact, fontAdrFacture));
		pptInfosFacture.addCell(cell);

		if (!id.paysFact.equals("")) {
			cell.setPhrase(new Phrase(id.paysFact, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}

		cell.setBorder(PdfPCell.BOX - PdfPCell.TOP);
		cell.setMinimumHeight(5);
		cell.setPhrase(new Phrase("", fontAdrFacture));
		pptInfosFacture.addCell(cell);

		PdfPTable pptGlobale = new PdfPTable(2);
		int[] widths = { 85, 15 };
		pptGlobale.setWidths(widths);

		PdfPCell cellGlob = new PdfPCell(pptInfosFacture);
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		cellGlob = new PdfPCell(pptGlobale.getDefaultCell());
		cellGlob.setPhrase(new Phrase(""));
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		return pptGlobale;
	}

	private PdfPTable makeAdrLivWeb() throws DocumentException {

		PdfPTable pptInfosLiv = new PdfPTable(1);

		PdfPCell cell = new PdfPCell(pptInfosLiv.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setBackgroundColor(new Color(175, 175, 175));

		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);

		cell.setPhrase(new Phrase("LIVRAISON", fontSmallTitre));
		pptInfosLiv.addCell(cell);

		cell.setBackgroundColor(new Color(255, 255, 255));

		cell.setPaddingBottom(0);
		cell.setPaddingLeft(15);

		cell.setBorder(PdfPCell.BOX - PdfPCell.BOTTOM);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.raisonSocialeDest, fontAdrFacture));
		pptInfosLiv.addCell(cell);

		cell.setBorder(PdfPCell.LEFT + PdfPCell.RIGHT);

		cell.setPhrase(new Phrase(id.adresse1Dest, fontAdrFacture));
		pptInfosLiv.addCell(cell);

		if (!id.adresse2Dest.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Dest, fontAdrFacture));
			pptInfosLiv.addCell(cell);
		}

		if (!id.adresse3Dest.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Dest, fontAdrFacture));
			pptInfosLiv.addCell(cell);
		}

		cell.setPhrase(new Phrase(id.codePostalDest + " " + id.villeDest, fontAdrFacture));
		pptInfosLiv.addCell(cell);

		if (!id.paysDest.equals("")) {
			cell.setPhrase(new Phrase(id.paysDest, fontAdrFacture));
			pptInfosLiv.addCell(cell);
		}

		cell.setBorder(PdfPCell.BOX - PdfPCell.TOP);
		cell.setMinimumHeight(5);
		cell.setPhrase(new Phrase("", fontAdrFacture));
		pptInfosLiv.addCell(cell);

		PdfPTable pptGlobale = new PdfPTable(2);
		int[] widths = { 15, 85 };
		pptGlobale.setWidths(widths);

		PdfPCell cellGlob = new PdfPCell(pptGlobale.getDefaultCell());
		cellGlob.setPhrase(new Phrase(""));
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		cellGlob = new PdfPCell(pptInfosLiv);
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		return pptGlobale;
	}

	
	private PdfPTable makeInfosSociete() {

		PdfPTable pptInfos = new PdfPTable(1);

		PdfPCell cell;

		if (logo != null) {
			cell = new PdfPCell(logo, false);

			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

			pptInfos.addCell(cell);

			cell.setImage(null);

			cell.setMinimumHeight(5);
			cell.setPhrase(new Phrase("", fontAdrSociete));
			pptInfos.addCell(cell);
			cell.setMinimumHeight(0);
		}
		else {
			cell = new PdfPCell(pptInfos.getDefaultCell());
			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		}		

		if (!id.logoAdr) {
			cell.setPhrase(new Phrase(id.raisonSociale, fontAdrSocieteGras));		
			pptInfos.addCell(cell);
			cell.setPhrase(new Phrase(id.adresse1, fontAdrSociete));		
			pptInfos.addCell(cell);
			if (!id.adresse2.equals("")) {
				cell.setPhrase(new Phrase(id.adresse2, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			if (!id.adresse3.equals("")) {
				cell.setPhrase(new Phrase(id.adresse3, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			cell.setPhrase(new Phrase(id.codePostal+" "+id.ville, fontAdrSociete));		
			pptInfos.addCell(cell);

			String telfax = "";

			if (id.tel.length()>0) {
				telfax = "T\u00E9l: "+ id.tel;
				if (id.fax.length()>0) {
					telfax += " - Fax: "+ id.fax;
				}
			}
			else if (id.fax.length()>0) {
				telfax = "Fax: "+ id.fax;
			}		

			if (telfax.length()>0) {
				cell.setPhrase(new Phrase(telfax, fontAdrSociete));		
				pptInfos.addCell(cell);
			}

			if (!id.email.equals("")) {
				cell.setPhrase(new Phrase("e-mail: "+ id.email, fontAdrSociete));		
				pptInfos.addCell(cell);
			}
			if (!id.web.equals("")) {
				cell.setPhrase(new Phrase(id.web, fontWeb));		
				pptInfos.addCell(cell);
			}
		}

		cell.setPhrase(new Phrase(""));		
		pptInfos.addCell(cell);

		return pptInfos;	
	}


	private PdfPTable makeInfosFacture() throws DocumentException {
		
		PdfPTable pptInfosFacture = new PdfPTable(2);
		
		int[] widths = {60,40};
		pptInfosFacture.setWidths(widths);
		
		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingLeft(4);
		cell.setPaddingBottom(4);
		
		cell.setPhrase(new Phrase("BP du BL N\u00B0 "+ id.numeroBon, fontInfosFact));
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase("Date : "+ formatDate.format(new Date(id.dateBon)), fontInfosFact));
		pptInfosFacture.addCell(cell);
		
		cell.setColspan(2);
		if (id.clientId.length()>0) {
			cell.setPhrase(new Phrase("Client : "+ id.clientId +" - "+ id.raisonSocialeFact, fontInfosFact));
		}
		else {
			cell.setPhrase(new Phrase("Client : "+ id.raisonSocialeFact, fontInfosFact));
		}
		pptInfosFacture.addCell(cell);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(5);
		pptInfosFacture.addCell(cell);
		cell.setMinimumHeight(0);
		
		cell.setPhrase(new Phrase("Page "+currentPage+" / "+totalPages, fontInfosFact));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(55);
		pptInfosFacture.addCell(cell);
		
		return pptInfosFacture;		
	}
	
	
	private PdfPTable makeAdrLiv() throws DocumentException {

		PdfPTable pptInfosFacture = new PdfPTable(2);

		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());
		cell.setColspan(2);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setMinimumHeight(0);
		cell.setPaddingLeft(0);
		cell.setPaddingBottom(0);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.raisonSocialeDest, fontAdrFacture));
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase(id.adresse1Dest, fontAdrFacture));
		pptInfosFacture.addCell(cell);
		
		if (!id.adresse2Dest.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Dest, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}
		
		if (!id.adresse3Dest.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Dest, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setPhrase(new Phrase(id.codePostalDest +" "+ id.villeDest, fontAdrFacture));
		pptInfosFacture.addCell(cell);
		
		if (!id.paysDest.equals("")) {
			cell.setPhrase(new Phrase(id.paysDest, fontAdrFacture));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("", fontLine));
		pptInfosFacture.addCell(cell);

		PdfPTable pptGlobale = new PdfPTable(2);
		int[] widths = {85,15};
		pptGlobale.setWidths(widths);

		PdfPCell cellGlob = new PdfPCell(pptInfosFacture);
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		cellGlob = new PdfPCell(pptGlobale.getDefaultCell());
		cellGlob.setPhrase(new Phrase(""));
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		return pptGlobale;
	}


	private PdfPTable makeTitre() {
	
		PdfPTable pptTitre = new PdfPTable(1);
		pptTitre.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptTitre.getDefaultCell());		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setFixedHeight(30);
		cell.setPaddingBottom(5);
		cell.setPhrase(new Phrase("B  O  N     D  E     P  R  E  P  A  R  A  T  I  O  N", fontTitre));
		
		pptTitre.addCell(cell);
		
		return pptTitre;	
	}
	
	
	private PdfPTable makeResponsable() throws DocumentException {
		
		boolean inter = (id.interlocuteurLiv.length()>0);

		PdfPTable pptResponsable = new PdfPTable(2);
		int[] widthsCadreResp = {30,70};
		pptResponsable.setWidths(widthsCadreResp);
		pptResponsable.setWidthPercentage(100);

		PdfPTable pptCadreGauche = new PdfPTable(1);
		pptCadreGauche.setWidthPercentage(100);
		
		PdfPTable pptCadreDroit = new PdfPTable(2);
		int[] widthsCadreDroit = {50,50};
		pptCadreDroit.setWidths(widthsCadreDroit);
		pptCadreDroit.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptResponsable.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setPaddingLeft(5);
		cell.setBorder(PdfPCell.BOX-PdfPCell.BOTTOM);
		cell.setMinimumHeight(0);
		if (inter) {
			cell.setPhrase(new Phrase(""));
			pptResponsable.addCell(cell);
		} else {
			cell.setColspan(2);
		}
		cell.setPhrase(new Phrase(""));
		pptResponsable.addCell(cell);
		
		cell = new PdfPCell(pptCadreGauche.getDefaultCell());
		cell.setPaddingLeft(5);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase("R\u00E9f. Affaire : "+ id.refAffaire, fontLegale));
		pptCadreGauche.addCell(cell);
		cell.setPhrase(new Phrase("N\u00B0 Commande : "+ id.numCommande, fontLegale));
		pptCadreGauche.addCell(cell);
		cell.setPhrase(new Phrase("R\u00E9f. Commande : "+ id.refCommande, fontLegale));
		pptCadreGauche.addCell(cell);
		if (id.existeSites) {
			cell.setPhrase(new Phrase("Provenance : "+ id.provenance, fontLegale));
			pptCadreGauche.addCell(cell);
		}
		if (!id.modeExpedition.equals("")) {
			Phrase Liv=new Phrase("Livraison : ", fontLegale);
			Liv.add(new Chunk(id.modeExpedition, fontCadre1));
			cell.setPhrase(Liv);
			pptCadreGauche.addCell(cell);
		}
		
		if (inter) {
			cell = new PdfPCell(pptCadreDroit.getDefaultCell());
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
			cell.setPaddingLeft(5);
			cell.setBorder(PdfPCell.NO_BORDER);
			
			cell.setFixedHeight(20);
			cell.setPhrase(new Phrase("Contact : "+ id.interlocuteurLiv, fontLegale));
			pptCadreDroit.addCell(cell);
			cell.setMinimumHeight(0);
			cell.setPhrase(new Phrase("E-Mail : "+ id.emailInterLiv, fontLegale));
			pptCadreDroit.addCell(cell);
			
			cell.setFixedHeight(20);
			cell.setPhrase(new Phrase("T\u00E9l : "+ id.telInterLiv, fontLegale));
			pptCadreDroit.addCell(cell);
			cell.setMinimumHeight(0);
			cell.setPhrase(new Phrase("Fax : "+ id.faxInterLiv, fontLegale));
			pptCadreDroit.addCell(cell);
		}
		
		cell = new PdfPCell(pptCadreGauche);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		if (!inter) {
			cell.setColspan(2);
		}
		pptResponsable.addCell(cell);
		
		if (inter) {
			cell = new PdfPCell(pptCadreDroit);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
			cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
			pptResponsable.addCell(cell);
		}
		
		cell = new PdfPCell(pptResponsable.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setPaddingLeft(5);
		cell.setBorder(PdfPCell.BOX-PdfPCell.TOP);
		cell.setMinimumHeight(0);
		if (inter) {
			cell.setPhrase(new Phrase(""));
			pptResponsable.addCell(cell);
		} else {
			cell.setColspan(2);
		}
		cell.setPhrase(new Phrase(""));
		pptResponsable.addCell(cell);

		return pptResponsable;
	}	
	
	
	private PdfPTable makeEntetesColonnes() throws DocumentException {

		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingBottom(4);
		cell.setMinimumHeight(40);

		cell.setPhrase(new Phrase("Localisation", fontEnteteColonne));
		pptCols.addCell(cell);
		cell.setPhrase(new Phrase("R\u00E9f\u00E9rence", fontEnteteColonne));
		pptCols.addCell(cell);
		
		if (id.referenceFabricant) {
			cell.setPhrase(new Phrase("R\u00E9f\u00E9rence\nfabricant", fontEnteteColonne));
			pptCols.addCell(cell);			
		}
		
		cell.setPhrase(new Phrase("D\u00E9signation", fontEnteteColonne));
		pptCols.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase("Coul.", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		if (tail) {
			cell.setPhrase(new Phrase("Taille", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		if (numLot) {
			cell.setPhrase(new Phrase("N\u00B0 Lot", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		if (nbPieces) {
			cell.setPhrase(new Phrase("Nb Pc", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		if (datePeremption) {
			cell.setPhrase(new Phrase("DLC", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		cell.setPhrase(new Phrase("Qt\u00E9", fontEnteteColonne));
		pptCols.addCell(cell);
		
		if (unite) {
			cell.setPhrase(new Phrase("Unit\u00E9", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		if (id.stockReel) {
			cell.setPhrase(new Phrase("Stock\nr\u00E9el", fontEnteteColonne));
			pptCols.addCell(cell);	
		}
		if (id.stockDispo) {
			cell.setPhrase(new Phrase("Stock\ndispo", fontEnteteColonne));
			pptCols.addCell(cell);			
		}

		return pptCols;
	}	
	
	
	private PdfPTable makeLineArticle(LigneBonPreparation lb) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		
		if (lb.commentaireAvant.length()>0) {
			cell.setPaddingLeft(5);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPaddingRight(0);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);pptLine.addCell(cell);
			if (id.referenceFabricant) pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lb.commentaireAvant, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
			if (id.stockReel) pptLine.addCell(cell);
			if (id.stockDispo) pptLine.addCell(cell);
		}

		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase(lb.localisation, fontLine));
		pptLine.addCell(cell);
		cell.setPhrase(new Phrase(lb.reference, fontLine));
		pptLine.addCell(cell);
		if (id.referenceFabricant) {
			cell.setPhrase(new Phrase(lb.refFabricant, fontLine));
			pptLine.addCell(cell);
		}	
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(lb.designation, fontLine));
		pptLine.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase(lb.couleur, fontLine));
			pptLine.addCell(cell);
		}
		if (tail) {
			cell.setPhrase(new Phrase(lb.taille, fontLine));
			pptLine.addCell(cell);
		}
		
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		
		if (numLot) {
			cell.setPhrase(new Phrase((lb.num_lot), fontLine));
			pptLine.addCell(cell);
		}
		if (nbPieces) {
			if (lb.nb_pieces != 0)
				cell.setPhrase(new Phrase(""+numFormatCap.format(lb.nb_pieces), fontLine));
			else
				cell.setPhrase(new Phrase("", fontLine));			
			pptLine.addCell(cell);
		}
		if (datePeremption) {
			if (lb.date_peremption != 0)
				cell.setPhrase(new Phrase(""+ formatDate.format(new Date(lb.date_peremption)), fontLine));
			else
				cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
		}

		cell.setPhrase(new Phrase(""+numFormatCap.format(lb.quantite), fontLine));
		pptLine.addCell(cell);
		
		if (unite) {
			cell.setPhrase(new Phrase(lb.unite, fontLine));
			pptLine.addCell(cell);
		}
		
		if (id.stockReel) {
			cell.setPhrase(new Phrase(""+numFormatCap.format(lb.stockReel), fontLine));
			pptLine.addCell(cell);
		}
		if (id.stockDispo) {
			cell.setPhrase(new Phrase(""+numFormatCap.format(lb.stockDispo), fontLine));
			pptLine.addCell(cell);
		}	


		if (lb.commentaire.length()>0) {
			cell.setPaddingLeft(5);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPaddingRight(0);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);pptLine.addCell(cell);
			if (id.referenceFabricant) pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lb.commentaire, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
			if (id.stockReel) pptLine.addCell(cell);
			if (id.stockDispo) pptLine.addCell(cell);
		}

		if (lb.detail_1.length()>0) {
			cell.setPaddingLeft(5);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPaddingRight(0);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);pptLine.addCell(cell);
			if (id.referenceFabricant) pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lb.detail_1, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
			if (id.stockReel) pptLine.addCell(cell);
			if (id.stockDispo) pptLine.addCell(cell);
		}

		if (lb.detail_2.length()>0) {
			cell.setPaddingLeft(5);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPaddingRight(0);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);pptLine.addCell(cell);
			if (id.referenceFabricant) pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lb.detail_2, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
			if (id.stockReel) pptLine.addCell(cell);
			if (id.stockDispo) pptLine.addCell(cell);
		}		

		return pptLine;	
	}
	
	
	private PdfPTable makeLineNomenclature(LigneBonPreparation lb) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		if (id.referenceFabricant) pptLine.addCell(cell);

		PdfPTable pptArtNom = new PdfPTable(unite?3:2);
		pptArtNom.setWidthPercentage(100);
		int[] widths = {10,90};
		int[] widthsUnite = {10,10,80};
		pptArtNom.setWidths(unite?widthsUnite:widths);

		PdfPCell cellArtNom = new PdfPCell(pptArtNom.getDefaultCell());
		cellArtNom.setBorder(PdfPCell.NO_BORDER);

		cellArtNom.setPhrase(new Phrase(numFormatCap.format(lb.quantite), fontArtNom));
		pptArtNom.addCell(cellArtNom);
		
		if (unite) {
			cellArtNom.setPhrase(new Phrase(lb.unite, fontArtNom));
			pptArtNom.addCell(cellArtNom);
		}

		cellArtNom.setPhrase(new Phrase(lb.designation, fontArtNom));
		pptArtNom.addCell(cellArtNom);


		PdfPCell cellDesc = new PdfPCell(pptArtNom);
		cellDesc.setBorder(PdfPCell.NO_BORDER);
		cellDesc.setPaddingLeft(10);
		pptLine.addCell(cellDesc);

		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		if (coul) pptLine.addCell(cell);
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);
		if (id.stockReel) pptLine.addCell(cell);
		if (id.stockDispo) pptLine.addCell(cell);

		return pptLine;
	}
	
	
	private PdfPTable makeLineCommentaire(LigneBonPreparation lb) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		if (id.referenceFabricant) pptLine.addCell(cell);
		cell.setPhrase(new Phrase(lb.designation, fontLine));
		pptLine.addCell(cell);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		if (coul) pptLine.addCell(cell);
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);		
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);
		if (id.stockReel) pptLine.addCell(cell);
		if (id.stockDispo) pptLine.addCell(cell);

		return pptLine;	
	}
	
	
	private PdfPTable makeFinTableau() throws DocumentException {
	
		PdfPTable pptFin = new PdfPTable(1);
		pptFin.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptFin.getDefaultCell());
		cell.setPhrase(new Phrase(""));
		cell.setFixedHeight(1);
		cell.setBorder(PdfPCell.TOP);
		cell.setPadding(0);
		pptFin.addCell(cell);
		
		return pptFin;
	}
	
	
	private PdfPTable makeZoneCommentaire() throws DocumentException {		
		
		PdfPTable pptCom = new PdfPTable(nbCols);
		pptCom.setWidthPercentage(100);
		pptCom.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptCom.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT + PdfPCell.RIGHT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("", fontLine));
		pptCom.addCell(cell);
		pptCom.addCell(cell);
		if (id.referenceFabricant) pptCom.addCell(cell);
		cell.setPhrase(new Phrase(id.comFin + id.mentions, fontLine));
		pptCom.addCell(cell);
		cell.setPhrase(new Phrase("", fontLine));
		pptCom.addCell(cell);
		
		if (id.stockReel) pptCom.addCell(cell);
		if (id.stockDispo) pptCom.addCell(cell);
		if (coul) pptCom.addCell(cell);		
		if (tail) pptCom.addCell(cell);
		if (unite) pptCom.addCell(cell);		
		if (numLot) pptCom.addCell(cell);
		if (nbPieces) pptCom.addCell(cell);
		if (datePeremption) pptCom.addCell(cell);
	
		return pptCom;		
	}
	
	
	private PdfPTable makeInfosLegales() {
	
		PdfPTable pptLegales = new PdfPTable(1);
		pptLegales.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptLegales.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(id.raisonSociale +" - "+ id.typeSociete + (id.capital>0?" au capital de "+ numFormatCap.format(id.capital) +" Euros":"") +" - immatricul\u00E9e au RCS "+ id.rcs +" "+ id.siren +" - N\u00B0 TVA : "+ id.tvaIntra, fontLegale));
		
		pptLegales.addCell(cell);
		
		return pptLegales;	
	}
	

} // fin BonPreparationStd
