/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration d'un bon de reliquats au format standard
 */

package org.opensi.facturation.actions.documents.modeles;


import java.awt.Color;
import java.io.IOException;
import java.sql.Date;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;

import org.opensi.util.pdf.MultiPagesDocument;
import org.opensi.util.pdf.PdfWidths;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class BonReliquatsStd extends MultiPagesDocument {


private static final Font fontAdrFacture = new Font(Font.HELVETICA, 11, Font.BOLD);
private static final Font fontAdrSociete = new Font(Font.HELVETICA, 10, Font.NORMAL);
private static final Font fontAdrSocieteGras = new Font(Font.HELVETICA, 11, Font.BOLD);
private static final Font fontTitre = new Font(Font.HELVETICA , 16, Font.BOLD, Color.WHITE);
private static final Font fontModalites = new Font(Font.HELVETICA , 10, Font.BOLD);
private static final Font fontWeb = new Font(Font.HELVETICA, 10, Font.ITALIC);
private static final Font fontEnteteColonne = new Font(Font.HELVETICA, 9, Font.BOLD);
private static final Font fontInfosFact = new Font(Font.HELVETICA , 10, Font.NORMAL);
private static final Font fontLegale = new Font(Font.HELVETICA , 7, Font.NORMAL);
private static final Font fontLine = new Font(Font.HELVETICA , 8, Font.NORMAL);
private static final Font fontTotaux = new Font(Font.HELVETICA , 10, Font.NORMAL);
private static final Font fontArtNom = new Font(Font.HELVETICA, 9, Font.ITALIC);


private DecimalFormat numFormat = new DecimalFormat("0.00");
private DecimalFormat numFormat4 = new DecimalFormat("0.00##");
private DecimalFormat numFormatCap = new DecimalFormat("0.##");
private SimpleDateFormat formatDate = new SimpleDateFormat("dd/MM/yyyy");


private DataBonReliquats id;	   // les donnes dynamiques du bon de reliquats

private float[] widthsCols;
private int nbCols;

private boolean rist;
private boolean coul;
private boolean unite;

private boolean chiffre = true;

private double sousTotalCourant = 0;

private Image logo;



  public BonReliquatsStd(String tempdir, DataBonReliquats id, boolean chiffre) throws Exception {

		super(tempdir);
		
		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');
		
		numFormat.setDecimalFormatSymbols(dfs);
		numFormat.setNegativeSuffix("");
		numFormat.setNegativePrefix("-");
		numFormat.setPositiveSuffix(" ");
		numFormat.setGroupingUsed(true);
		numFormat.setGroupingSize(3);		
		
		numFormat4.setDecimalFormatSymbols(dfs);
		numFormat4.setGroupingUsed(true);
		numFormat4.setGroupingSize(3);
		
		numFormatCap.setDecimalFormatSymbols(dfs);
		numFormatCap.setGroupingUsed(true);
		numFormatCap.setGroupingSize(3);		
		
		this.id = id;
		this.chiffre = chiffre;
		
		try {
			logo = Image.getInstance(id.urlLogo);
			if (logo.getDpiX()!=0 && logo.getDpiY()!=0)
				logo.scaleAbsolute(logo.getPlainWidth() * 72 / logo.getDpiX(), logo.getPlainHeight() * 72 / logo.getDpiY());
		}
		catch (IOException e) {
			logo = null;
		}
		
		rist = id.rist;
		coul = id.coul;
		unite = id.unite;
		
		PdfWidths pw = new PdfWidths();
		
		pw.addColonneVariableMinMax(2, 12, 20);
		pw.addColonneVariableMin(4,20);
		if (coul) pw.addColonneVariableMin(1, 6);
		pw.addColonneFixe(8);
		pw.addColonneFixe(9);
		pw.addColonneVariableMin(1, 8);
		pw.addColonneVariableMin(1, 8);
		if (unite) pw.addColonneFixe(5);
		if (chiffre) {
			pw.addColonneVariableMin(1, 6);
			if (rist) pw.addColonneFixe(6);
			pw.addColonneVariableMin(1, 9);
		}
		
		nbCols = pw.getNbCols();
		widthsCols = pw.getWidthsCols();
	}


	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		tb.add(makeEntetesColonnes());		
		
		return tb.getTable();	
	}	
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(firstHeader());
		
		if (chiffre) {
			tb.add(makeSousTotal(id.calcSousTotal(lastLine)));
		}
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeTotauxLast());		
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());
				
		return tb.getTable();
	}	
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
	
		sousTotalCourant = id.calcSousTotal(lastLine);
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(emptyLine(10));
		tb.add(makeTotauxMiddle());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());		
		
		return tb.getTable();
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		LigneBonReliquats lc = id.lignes.get(numLine);
	
		switch (lc.type) {
			case 'A': return makeLineArticle(lc);
			case 'N': 
			default: return makeLineNomenclature(lc);
		}		
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {
		
		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));		
	
		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}
	
		return pptSpace;
	}
	
	
	public int nbLines() {
	
		return id.getNbLignes();
	}
	
	
	public int nbEx() {
	
		return 1;
	}


	public Image imageFond() {
		return null;
	}


	/* mthodes internes */

	private PdfPTable makeSpace(int space) {
	
		PdfPTable pptSpace = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));
	
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	
	
	private PdfPTable makeEntete() throws DocumentException {

		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		
		PdfPTable pptEnteteGauche = new PdfPTable(1);
		pptEnteteGauche.setWidthPercentage(100);
		PdfPTable pptEnteteDroite = new PdfPTable(1);
		pptEnteteDroite.setWidthPercentage(100);
		
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setMinimumHeight(115);
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		pptEnteteGauche.addCell(cellSociete);
		PdfPCell cellVide = new PdfPCell(pptEnteteGauche.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		pptEnteteGauche.addCell(cellVide);
		
		PdfPCell cellFacture = new PdfPCell(makeInfosBon());
		cellFacture.setFixedHeight(115);
		cellFacture.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellFacture);		
		PdfPCell cellAdrEnvoi = new PdfPCell(makeAdrLiv());
		cellAdrEnvoi.setMinimumHeight(80);
		cellAdrEnvoi.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellAdrEnvoi);
		pptEnteteDroite.addCell(cellVide);
		
		PdfPCell cellPartieGauche = new PdfPCell(pptEnteteGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellPartieDroite = new PdfPCell(pptEnteteDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		pptEntete.addCell(cellPartieGauche);
		pptEntete.addCell(cellPartieDroite);
		
		return pptEntete;	
	}
	

	private PdfPTable makeInfosSociete() throws DocumentException {
	
		PdfPTable pptInfos = new PdfPTable(1);
		
		PdfPCell cell;
		
		if (logo != null) {
			cell = new PdfPCell(logo, false);

			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

			pptInfos.addCell(cell);

			cell.setImage(null);

			cell.setMinimumHeight(5);
			cell.setPhrase(new Phrase("", fontAdrSociete));
			pptInfos.addCell(cell);
			cell.setMinimumHeight(0);
		}
		else {
			cell = new PdfPCell(pptInfos.getDefaultCell());
			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		}		
		
		if (!id.logoAdr) {
			cell.setPhrase(new Phrase(id.raisonSociale, fontAdrSocieteGras));		
			pptInfos.addCell(cell);
			cell.setPhrase(new Phrase(id.adresse1, fontAdrSociete));		
			pptInfos.addCell(cell);
			if (!id.adresse2.equals("")) {
				cell.setPhrase(new Phrase(id.adresse2, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			if (!id.adresse3.equals("")) {
				cell.setPhrase(new Phrase(id.adresse3, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			cell.setPhrase(new Phrase(id.codePostal+" "+id.ville, fontAdrSociete));		
			pptInfos.addCell(cell);

			String telfax = "";

			if (id.tel.length()>0) {
				telfax = "T\u00E9l: "+ id.tel;
				if (id.fax.length()>0) {
					telfax += " - Fax: "+ id.fax;
				}
			}
			else if (id.fax.length()>0) {
				telfax = "Fax: "+ id.fax;
			}		

			if (telfax.length()>0) {
				cell.setPhrase(new Phrase(telfax, fontAdrSociete));		
				pptInfos.addCell(cell);
			}

			if (!id.email.equals("")) {
				cell.setPhrase(new Phrase("e-mail: "+ id.email, fontAdrSociete));		
				pptInfos.addCell(cell);
			}
			if (!id.web.equals("")) {
				cell.setPhrase(new Phrase(id.web, fontWeb));		
				pptInfos.addCell(cell);
			}
		}
		
		cell.setPhrase(new Phrase(""));		
		pptInfos.addCell(cell);
		
		return pptInfos;	
	}
	
	
	private PdfPTable makeAdrLiv() throws DocumentException {

		PdfPTable pptInfosBon = new PdfPTable(2);

		PdfPCell cell = new PdfPCell(pptInfosBon.getDefaultCell());
		cell.setColspan(2);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setMinimumHeight(0);
		cell.setPaddingLeft(0);
		cell.setPaddingBottom(0);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.raisonSocialeFourn, fontAdrFacture));
		pptInfosBon.addCell(cell);
		
		cell.setPhrase(new Phrase(id.adresse1Fourn, fontAdrFacture));
		pptInfosBon.addCell(cell);
		
		if (!id.adresse2Fourn.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Fourn, fontAdrFacture));
			pptInfosBon.addCell(cell);
		}
		
		if (!id.adresse3Fourn.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Fourn, fontAdrFacture));
			pptInfosBon.addCell(cell);
		}
		
		cell.setPhrase(new Phrase(id.codePostalFourn +" "+ id.villeFourn, fontAdrFacture));
		pptInfosBon.addCell(cell);
		
		if (!id.paysFourn.equals("")) {
			cell.setPhrase(new Phrase(id.paysFourn, fontAdrFacture));
			pptInfosBon.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("", fontLine));
		pptInfosBon.addCell(cell);

		PdfPTable pptGlobale = new PdfPTable(2);
		int[] widths = {85,15};
		pptGlobale.setWidths(widths);

		PdfPCell cellGlob = new PdfPCell(pptInfosBon);
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		cellGlob = new PdfPCell(pptGlobale.getDefaultCell());
		cellGlob.setPhrase(new Phrase(""));
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		return pptGlobale;
	}


	private PdfPTable makeInfosBon() throws DocumentException {
		
		PdfPTable pptInfosBon = new PdfPTable(1);
		pptInfosBon.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptInfosBon.getDefaultCell());
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingLeft(4);
		cell.setPaddingBottom(4);
		
		cell.setPhrase(new Phrase("Date : "+ formatDate.format(new Date(id.dateBonReliquats)), fontInfosFact));
		pptInfosBon.addCell(cell);
		
		String codeFournisseur = (id.fournisseurId.length()>0?"Fournisseur N\u00B0 "+ id.fournisseurId +" - ":"Fournisseur : ")+ id.raisonSocialeFourn;
		
		cell.setPhrase(new Phrase(codeFournisseur, fontInfosFact));
		pptInfosBon.addCell(cell);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(5);
		pptInfosBon.addCell(cell);
		cell.setMinimumHeight(0);
		
		cell.setPhrase(new Phrase("Page "+ currentPage +" / "+ totalPages, fontInfosFact));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		pptInfosBon.addCell(cell);
		
		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(55);
		pptInfosBon.addCell(cell);
		
		return pptInfosBon;		
	}
	


	private PdfPTable makeTitre() {
	
		PdfPTable pptTitre = new PdfPTable(1);
		pptTitre.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptTitre.getDefaultCell());		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setFixedHeight(30);
		cell.setPaddingBottom(5);
		cell.setPhrase(new Phrase("B  O  N     D  E     R  E  L  I  Q  U  A  T  S", fontTitre));
		
		pptTitre.addCell(cell);
		
		return pptTitre;	
	}
	
	
	private PdfPTable makeEntetesColonnes() throws DocumentException {

		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingBottom(4);
		cell.setMinimumHeight(40);

		cell.setPhrase(new Phrase("R\u00E9f\u00E9rence", fontEnteteColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("D\u00E9signation", fontEnteteColonne));
		pptCols.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase("Coul.", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("Date Cde", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("N\u00B0 Cde", fontEnteteColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("Quantit\u00E9\ncd\u00E9e", fontEnteteColonne));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("Reliquat", fontEnteteColonne));
		pptCols.addCell(cell);
		
		if (unite) {
			cell.setPhrase(new Phrase("Unit\u00E9", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		if (chiffre) {
			cell.setPhrase(new Phrase("PU HT", fontEnteteColonne));
			pptCols.addCell(cell);

			if (rist) {
				cell.setPhrase(new Phrase("Rist. %", fontEnteteColonne));
				pptCols.addCell(cell);
			}		
		
			cell.setPhrase(new Phrase("Montant HT", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		return pptCols;
	}
	
	
	private PdfPTable makeLineArticle(LigneBonReliquats lc) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);

		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase(lc.reference, fontLine));
		pptLine.addCell(cell);		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(lc.designation, fontLine));
		pptLine.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase(lc.couleur, fontLine));
			pptLine.addCell(cell);
		}
		
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		
		cell.setPhrase(new Phrase(""+ formatDate.format(new Date(lc.dateCommande)), fontLine));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(lc.numCommande, fontLine));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(""+ numFormatCap.format(lc.quantiteCde), fontLine));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(""+ numFormatCap.format(lc.reliquat), fontLine));
		pptLine.addCell(cell);
		
		if (unite) {
			cell.setPhrase(new Phrase(lc.unite, fontLine));
			pptLine.addCell(cell);
		}

		cell.setPaddingRight(3);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		
		if (chiffre) {
			cell.setPhrase(new Phrase(numFormat4.format(lc.prixUnitaire), fontLine));
			pptLine.addCell(cell);
			if (rist) {
				if (lc.ristourne!=0) {
					cell.setPhrase(new Phrase(numFormat.format(lc.ristourne), fontLine));
				}
				else {
					cell.setPhrase(new Phrase("", fontLine));
				}
				pptLine.addCell(cell);
			}

			cell.setPhrase(new Phrase(numFormat.format(lc.montant), fontLine));		
			pptLine.addCell(cell);
		}		
		

		if (lc.detail_1.length()>0) {
			cell.setPaddingLeft(5);
			cell.setPaddingRight(0);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lc.detail_1, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			if (coul) pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (chiffre) {
				pptLine.addCell(cell);
				if (rist) pptLine.addCell(cell);
				pptLine.addCell(cell);
			}
		}

		if (lc.detail_2.length()>0) {
			cell.setPaddingLeft(5);
			cell.setPaddingRight(0);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lc.detail_2, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			if (coul) pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (chiffre) {
				pptLine.addCell(cell);
				if (rist) pptLine.addCell(cell);
				pptLine.addCell(cell);
			}
		}

		return pptLine;	
	}
	
	
	private PdfPTable makeLineNomenclature(LigneBonReliquats lc) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);
		
		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		
		
		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		
		PdfPTable pptArtNom = new PdfPTable(unite?3:2);
		pptArtNom.setWidthPercentage(100);
		int[] widths = {10,90};
		int[] widthsUnite = {10,10,80};
		pptArtNom.setWidths(unite?widthsUnite:widths);

		PdfPCell cellArtNom = new PdfPCell(pptArtNom.getDefaultCell());
		cellArtNom.setBorder(PdfPCell.NO_BORDER);

		cellArtNom.setPhrase(new Phrase(numFormatCap.format(lc.quantiteCde), fontArtNom));
		pptArtNom.addCell(cellArtNom);
		
		if (unite) {
			cellArtNom.setPhrase(new Phrase(lc.unite, fontArtNom));
			pptArtNom.addCell(cellArtNom);
		}

		cellArtNom.setPhrase(new Phrase(lc.designation, fontArtNom));
		pptArtNom.addCell(cellArtNom);
		
		PdfPCell cellDesc = new PdfPCell(pptArtNom);
		cellDesc.setBorder(PdfPCell.NO_BORDER);
		cellDesc.setPaddingLeft(10);
		pptLine.addCell(cellDesc);
		
		cell.setPhrase(new Phrase("", fontLine));
		if (coul) pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		if (chiffre) {
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			pptLine.addCell(cell);
		}
		
		return pptLine;
	}
	
	
	
	private PdfPTable makeSousTotal(double repriseHT) throws DocumentException {
	
		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT+PdfPCell.BOTTOM);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingBottom(6);
		cell.setPaddingTop(3);

		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPaddingRight(3);
		cell.setPhrase(new Phrase("Sous-total HT", fontTotaux));
		pptLine.addCell(cell);
		cell.setPaddingLeft(0);

		cell.setPhrase(new Phrase("", fontLine));
		if (coul) pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		if (chiffre) {
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(repriseHT), fontTotaux));
			pptLine.addCell(cell);
		}

		return pptLine;	
	}	
	
	
	private PdfPTable makeTotauxLast() throws DocumentException {
		
		PdfPTable pptTotaux = new PdfPTable(3);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {50,15,35};
		pptTotaux.setWidths(widths);				
				
		if (chiffre) {

			PdfPCell cell = new PdfPCell(makeCadreDivers());
			cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);
			pptTotaux.addCell(cell);

			cell = new PdfPCell(pptTotaux.getDefaultCell());
			cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);
			cell.setPhrase(new Phrase(""));
			pptTotaux.addCell(cell);

			cell = new PdfPCell(makeCadreTotaux());
			cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);
			pptTotaux.addCell(cell);
		}
		else {

			PdfPCell cell = new PdfPCell(makeCadreDivers());
			cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);
			cell.setPaddingBottom(3);
			pptTotaux.addCell(cell);

			cell = new PdfPCell(pptTotaux.getDefaultCell());
			cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);
			cell.setPhrase(new Phrase(""));
			pptTotaux.addCell(cell);

			cell = new PdfPCell(makeCadreSousTotal());
			cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
			cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);
			pptTotaux.addCell(cell);
		}		
				
		return pptTotaux;	
	}	
		
	
	private PdfPTable makeTotauxMiddle() throws DocumentException {

		PdfPTable pptTotaux = new PdfPTable(3);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {50,15,35};
		pptTotaux.setWidths(widths);

		PdfPCell cell = new PdfPCell(makeCadreDivers());
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);
		cell.setPaddingBottom(3);
		pptTotaux.addCell(cell);

		cell = new PdfPCell(pptTotaux.getDefaultCell());
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);
		cell.setPhrase(new Phrase(""));
		pptTotaux.addCell(cell);

		cell = new PdfPCell(makeCadreSousTotal());
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);
		pptTotaux.addCell(cell);

		return pptTotaux;
	}
	
	
	private PdfPTable makeCadreDivers() throws DocumentException {

		PdfPTable pptDivers = new PdfPTable(1);
		pptDivers.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptDivers.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPhrase(new Phrase("", fontModalites));
		cell.setMinimumHeight(70);
		pptDivers.addCell(cell);

		return pptDivers;
	}
	
		
	private PdfPTable makeCadreSousTotal() throws DocumentException {

		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setBorder(PdfPCell.NO_BORDER);

		if (chiffre) {
			cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			cell.setPhrase(new Phrase("Sous-total HT :", fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(sousTotalCourant), fontTotaux));
			pptCadreTotaux.addCell(cell);
		}

		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);

		return pptCadreTotaux;
	}
	
	
	private PdfPTable makeCadreTotaux() throws DocumentException {

		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		
		if (id.mRemise>0) {
			cell.setPhrase(new Phrase("Montant HT :", fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.montantBase) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);
			
			cell.setPhrase(new Phrase("Remise :", fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(""+ numFormat.format(id.mRemise) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);
		}

		cell.setPhrase(new Phrase("Total HT :", fontTotaux));
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(id.totalBase) +" \u20AC", fontTotaux));
		pptCadreTotaux.addCell(cell);

		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);

		return pptCadreTotaux;
	}
	
	
	
	private PdfPTable makeInfosLegales() {
	
		PdfPTable pptLegales = new PdfPTable(1);
		pptLegales.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptLegales.getDefaultCell());		
		
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(id.raisonSociale +" - "+ id.typeSociete + (id.capital>0?" au capital de "+ numFormatCap.format(id.capital) +" Euros":"") +" - immatricul\u00E9e au RCS "+ id.rcs +" "+ id.siren +" - N\u00B0 TVA : "+ id.tvaIntra, fontLegale));
		
		pptLegales.addCell(cell);
		
		return pptLegales;	
	}
	

} // fin BonReliquatsStd
