/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration d'une facture au format standard
 */

package org.opensi.facturation.actions.documents.modeles;


import java.awt.Color;
import java.io.IOException;
import java.sql.Date;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.Iterator;

import org.opensi.util.calcul.CalculVentilTVA;
import org.opensi.util.pdf.PdfWidths;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.Chunk;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class FactureStd extends FacturePDF {


	private static final Font fontAdrGras = new Font(Font.HELVETICA, 11, Font.BOLD);
	private static final Font fontAdrSociete = new Font(Font.HELVETICA, 10, Font.NORMAL);
	private static final Font fontAdrSocieteGras = new Font(Font.HELVETICA, 11, Font.BOLD);
	private static final Font fontAdresse = new Font(Font.HELVETICA, 8, Font.NORMAL);
	private static final Font fontAdrContact = new Font(Font.HELVETICA, 7, Font.NORMAL);
	private static final Font fontTitre = new Font(Font.HELVETICA , 16, Font.BOLD, Color.WHITE);
	private static final Font fontWeb = new Font(Font.HELVETICA, 10, Font.ITALIC);
	private static final Font fontCadre1 = new Font(Font.HELVETICA, 10, Font.BOLD);
	private static final Font fontCadre1Petit = new Font(Font.HELVETICA, 8, Font.NORMAL);
	private static final Font fontCadre1PetitGras = new Font(Font.HELVETICA, 8, Font.BOLD);
	private static final Font fontEnteteColonne = new Font(Font.HELVETICA, 9, Font.BOLD);
	private static final Font fontInfosFact = new Font(Font.HELVETICA , 10, Font.NORMAL);
	private static final Font fontLegale = new Font(Font.HELVETICA , 7, Font.NORMAL);
	private static final Font fontLine = new Font(Font.HELVETICA , 8, Font.NORMAL);	
	private static final Font fontAdrLCR = new Font(Font.HELVETICA, 8, Font.NORMAL);
	private static final Font fontAdrGrasLCR = new Font(Font.HELVETICA, 8, Font.BOLD);
	private static final Font fontLineLCR = new Font(Font.HELVETICA, 6, Font.NORMAL);
	private static final Font fontLineGrasLCR = new Font(Font.HELVETICA, 6, Font.BOLD);
	private static final Font fontVirement = new Font(Font.HELVETICA , 7, Font.NORMAL);
	private static final Font fontModalites = new Font(Font.HELVETICA , 10, Font.BOLD);
	private static final Font fontTotaux = new Font(Font.HELVETICA , 10, Font.NORMAL);
	private static final Font fontTTC = new Font(Font.HELVETICA , 10, Font.BOLD);
	private static final Font fontArtNom = new Font(Font.HELVETICA, 9, Font.ITALIC);
	private static final Font fontSmallTitre = new Font(Font.HELVETICA , 8, Font.BOLD+Font.ITALIC, Color.WHITE);


	private DecimalFormat numFormat = new DecimalFormat("0.00");
	private DecimalFormat numFormat4 = new DecimalFormat("0.00##");
	private DecimalFormat numFormatCap = new DecimalFormat("0.##");
	private SimpleDateFormat formatDate = new SimpleDateFormat("dd/MM/yyyy");


	//private DataFacture id;	   // les donnes dynamiques de la facture

	private float[] widthsCols;
	private int nbCols;

	private boolean rist;
	private boolean coul;
	private boolean tail;
	private boolean unite;
	private boolean nbPieces;
	private boolean numLot;
	private boolean datePeremption;

	//private boolean multi = false;

	private double sousTotalCourant = 0;

	private Image logo;
	private boolean afficherAdrFact;

	
	public FactureStd(String tempdir, DataFacture id) throws Exception {

		super(tempdir, id);

		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');

		numFormat.setDecimalFormatSymbols(dfs);
		numFormat.setNegativeSuffix("");
		numFormat.setNegativePrefix("-");
		numFormat.setPositiveSuffix(" ");
		numFormat.setGroupingUsed(true);
		numFormat.setGroupingSize(3);		

		numFormat4.setDecimalFormatSymbols(dfs);
		numFormat4.setGroupingUsed(true);
		numFormat4.setGroupingSize(3);

		numFormatCap.setDecimalFormatSymbols(dfs);
		numFormatCap.setGroupingUsed(true);
		numFormatCap.setGroupingSize(3);		

		//this.id = id;

		try {
			logo = Image.getInstance(id.urlLogo);
			if (logo.getDpiX()!=0 && logo.getDpiY()!=0) {
				logo.scaleAbsolute(logo.getPlainWidth() * 72 / logo.getDpiX(), logo.getPlainHeight() * 72 / logo.getDpiY());
			}
		}
		catch (Exception e) {
			logo = null;
		}
		
		this.afficherAdrFact = (!id.raisonSocialeDest.equals(id.denominationEnvoi) || !id.adresseDest.equals(id.adresse1Envoi) || !id.compAdresseDest.equals(id.adresse2Envoi) || !id.adresseDest3.equals(id.adresse3Envoi)
			|| !id.codePostalDest.equals(id.codePostalEnvoi) || !id.villeDest.equals(id.villeEnvoi) || !id.paysDest.equals(id.paysEnvoi));

		rist = id.rist;
		tail = id.tail;
		coul = id.coul;
		unite = id.unite;
		nbPieces = id.nbPieces;
		numLot = id.numLot;
		datePeremption = id.datePeremption;

		PdfWidths pw = new PdfWidths();
		
		pw.addColonneVariableMinMax(2, 12, 20);
		pw.addColonneVariableMin(4,20);
		if (coul) pw.addColonneVariableMin(1, 6);
		if (tail) pw.addColonneVariableMin(1, 6);
		if (numLot) pw.addColonneVariableMinMax(2, 6, 10);
		if (nbPieces) pw.addColonneFixe(5);
		if (datePeremption) pw.addColonneFixe(8);
		pw.addColonneVariableMin(1, 8);
		if (unite) pw.addColonneFixe(5);
		pw.addColonneVariableMin(1, 6);
		if (rist) pw.addColonneFixe(6);
		pw.addColonneVariableMin(1, 9);
		
		nbCols = pw.getNbCols();
		widthsCols = pw.getWidthsCols();

	}



	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		if (this.afficherAdrFact || id.imprimerAdrLiv) {
			tb.add(makeAdresses());
			tb.add(makeSpace(5));
		}
		if ((id.directe==1 && id.codeFournisseur.length()>0)
		   || (id.codeFournisseur.length()>0 || !id.refAffaire.equals("") || !id.refCommande.equals("") || !id.refBL.equals(""))) {
			tb.add(makeResponsable());
			tb.add(makeSpace(5));
		}
		tb.add(makeEntetesColonnes());
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(5));
		tb.add(makeTitre());
		tb.add(makeSpace(5));
		tb.add(makeEntetesColonnes());
		
		tb.add(makeSousTotal(id.calcSousTotal(lastLine)));
		
		return tb.getTable();
	}


	public PdfPTable lastFooter() throws DocumentException {

		TableBuilder tb = new TableBuilder();

		if (id.comFin.length()>0 || id.mentions.length()>0) {
			tb.add(makeZoneCommentaire());
			tb.add(emptyLine(10));
		}
		tb.add(makeTotauxLast());		
		tb.add(makeSpace(5));
		tb.add(makeCadreBas2());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());
		
		if (id.imprimerLCR) {
			tb.add(makeSpace(3));
			tb.add(makeLCR());
		} else if (id.imprimerRIB) {
			tb.add(makeSpace(3));
			tb.add(makeVirement());
		}

		return tb.getTable();
	}	


	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {

		sousTotalCourant = id.calcSousTotal(lastLine);

		TableBuilder tb = new TableBuilder();

		tb.add(emptyLine(10));
		tb.add(makeTotauxMiddle());
		tb.add(makeSpace(3));
		tb.add(makeInfosLegales());		

		return tb.getTable();		
	}


	public PdfPTable bodyLine(int numLine) throws DocumentException {

		LigneFacture lf = id.lignes.get(numLine);

		switch (lf.type) {			
		case 'A':	return makeLineArticle(lf);
		case 'N': return makeLineNomenclature(lf);
		case 'C':
		default: return makeLineCommentaire(lf);
		}		
	}


	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {

		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));

		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}

		return pptSpace;
	}


	public int nbLines() {

		return id.getNbLignes();
	}


	public int nbEx() {

		return (multi?id.getNbEx():1);
	}


	public Image imageFond() {
		if (id.apercu) {	
			try {
				Image fond = Image.getInstance(id.urlApercu);
				fond.scaleAbsolute(324, 468);
				fond.setAbsolutePosition(135,100);
				
				return fond;
			}
			catch (Exception e) {
				return null;
			}
		}
		else
			return null;
	}



	/* mthodes internes */

	private PdfPTable makeSpace(int space) {

		PdfPTable pptSpace = new PdfPTable(1);
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));

		pptSpace.addCell(cell);

		return pptSpace;
	}


	private PdfPTable makeEntete() throws DocumentException {
		
		PdfPTable pptEntete = new PdfPTable(2);
		pptEntete.setWidthPercentage(100);
		int[] widths = {50,50};
		pptEntete.setWidths(widths);
		
		PdfPTable pptEnteteGauche = new PdfPTable(1);
		pptEnteteGauche.setWidthPercentage(100);
		PdfPTable pptEnteteDroite = new PdfPTable(1);
		pptEnteteDroite.setWidthPercentage(100);
		
		PdfPCell cellSociete = new PdfPCell(makeInfosSociete());
		cellSociete.setMinimumHeight(115);
		cellSociete.setBorder(PdfPCell.NO_BORDER);
		pptEnteteGauche.addCell(cellSociete);
		PdfPCell cellVide = new PdfPCell(pptEnteteGauche.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		pptEnteteGauche.addCell(cellVide);
		
		PdfPCell cellFacture = new PdfPCell(makeInfosFacture());
		cellFacture.setFixedHeight(115);
		cellFacture.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellFacture);		
		PdfPCell cellAdrEnvoi = new PdfPCell(makeAdrEnvoi());
		cellAdrEnvoi.setMinimumHeight(80);
		cellAdrEnvoi.setBorder(PdfPCell.NO_BORDER);
		pptEnteteDroite.addCell(cellAdrEnvoi);
		pptEnteteDroite.addCell(cellVide);
		
		PdfPCell cellPartieGauche = new PdfPCell(pptEnteteGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellPartieDroite = new PdfPCell(pptEnteteDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		pptEntete.addCell(cellPartieGauche);
		pptEntete.addCell(cellPartieDroite);
		
		return pptEntete;		
	}
	
	
	private PdfPTable makeAdresses() throws DocumentException {
		
		PdfPTable pptAdresses = new PdfPTable(!this.afficherAdrFact || !id.imprimerAdrLiv?1:3);
		pptAdresses.setWidthPercentage(100);
		if (this.afficherAdrFact && id.imprimerAdrLiv) {
			int[] widths = {49,2,49};
			pptAdresses.setWidths(widths);
		}
		
		PdfPTable pptAdresseGauche = new PdfPTable(1);
		pptAdresseGauche.setWidthPercentage(100);
		PdfPTable pptAdresseDroite = new PdfPTable(1);
		pptAdresseDroite.setWidthPercentage(100);
		
		if (this.afficherAdrFact) {
			PdfPCell cellAdrFact = new PdfPCell(makeAdrFact());
			cellAdrFact.setBorder(PdfPCell.NO_BORDER);
			pptAdresseGauche.addCell(cellAdrFact);
		}
		if (id.imprimerAdrLiv) {
			PdfPCell cellAdrLiv = new PdfPCell(makeAdrLiv());
			cellAdrLiv.setBorder(PdfPCell.NO_BORDER);
			pptAdresseDroite.addCell(cellAdrLiv);
		}
		
		PdfPCell cellPartieGauche = new PdfPCell(pptAdresseGauche);
		cellPartieGauche.setBorder(PdfPCell.NO_BORDER);
		PdfPCell cellVide = new PdfPCell(pptAdresses.getDefaultCell());
		cellVide.setBorder(PdfPCell.NO_BORDER);
		cellVide.setPhrase(new Phrase(""));
		PdfPCell cellPartieDroite = new PdfPCell(pptAdresseDroite);		
		cellPartieDroite.setBorder(PdfPCell.NO_BORDER);
		if (this.afficherAdrFact) { pptAdresses.addCell(cellPartieGauche); }
		if (this.afficherAdrFact && id.imprimerAdrLiv) { pptAdresses.addCell(cellVide); }
		if (id.imprimerAdrLiv) { pptAdresses.addCell(cellPartieDroite); }
		
		return pptAdresses;
	}


	private PdfPTable makeInfosSociete() {
	
		PdfPTable pptInfos = new PdfPTable(1);
		
		PdfPCell cell;
				
		if (logo != null) {
			cell = new PdfPCell(logo, false);

			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

			pptInfos.addCell(cell);

			cell.setImage(null);

			cell.setMinimumHeight(5);
			cell.setPhrase(new Phrase("", fontAdrSociete));
			pptInfos.addCell(cell);
			cell.setMinimumHeight(0);
		}
		else {
			cell = new PdfPCell(pptInfos.getDefaultCell());
			cell.setBorder(PdfPCell.NO_BORDER);
			cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		}	
		
		if (!id.logoAdr) {
			cell.setPhrase(new Phrase(id.raisonSociale, fontAdrSocieteGras));		
			pptInfos.addCell(cell);
			cell.setPhrase(new Phrase(id.adresse1, fontAdrSociete));		
			pptInfos.addCell(cell);
			if (!id.adresse2.equals("")) {
				cell.setPhrase(new Phrase(id.adresse2, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			if (!id.adresse3.equals("")) {
				cell.setPhrase(new Phrase(id.adresse3, fontAdrSociete));
				pptInfos.addCell(cell);
			}
			cell.setPhrase(new Phrase(id.codePostal+" "+id.ville, fontAdrSociete));		
			pptInfos.addCell(cell);

			String telfax = "";

			if (id.tel.length()>0) {
				telfax = "T\u00E9l: "+ id.tel;
				if (id.fax.length()>0) {
					telfax += " - Fax: "+ id.fax;
				}
			}
			else if (id.fax.length()>0) {
				telfax = "Fax: "+ id.fax;
			}		

			if (telfax.length()>0) {
				cell.setPhrase(new Phrase(telfax, fontAdrSociete));		
				pptInfos.addCell(cell);
			}

			if (!id.email.equals("")) {
				cell.setPhrase(new Phrase("e-mail: "+ id.email, fontAdrSociete));		
				pptInfos.addCell(cell);
			}
			if (!id.web.equals("")) {
				cell.setPhrase(new Phrase(id.web, fontWeb));		
				pptInfos.addCell(cell);
			}
		}
		
		cell.setPhrase(new Phrase(""));		
		pptInfos.addCell(cell);
		
		return pptInfos;	
	}
	
	
	private PdfPTable makeAdrFact() throws DocumentException {
		
		PdfPTable pptAdresses = new PdfPTable(id.interlocuteur.equals("")?1:2);
		pptAdresses.setWidthPercentage(100);
		if (!id.interlocuteur.equals("")) {
			int[] widths = {50,50};
			pptAdresses.setWidths(widths);
		}
		
		PdfPCell cell = new PdfPCell(pptAdresses.getDefaultCell());
		if (!id.interlocuteur.equals("")) { cell.setColspan(2); }
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
		cell.setPhrase(new Phrase("FACTURATION", fontSmallTitre));
		pptAdresses.addCell(cell);
		
		PdfPTable pptAdresseGauche = new PdfPTable(1);
		pptAdresseGauche.setWidthPercentage(100);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(Color.WHITE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.raisonSocialeDest, fontAdresse));
		pptAdresseGauche.addCell(cell);
		cell.setPhrase(new Phrase(id.adresseDest, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.compAdresseDest.equals("")) {
			cell.setPhrase(new Phrase(id.compAdresseDest, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		if (!id.adresseDest3.equals("")) {
			cell.setPhrase(new Phrase(id.adresseDest3, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		cell.setPhrase(new Phrase(id.codePostalDest+" "+id.villeDest, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.paysDest.equals("")) {
			cell.setPhrase(new Phrase(id.paysDest, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		
		PdfPCell cellPartieGauche = new PdfPCell(pptAdresseGauche);
		pptAdresses.addCell(cellPartieGauche);
		
		if (!id.interlocuteur.equals("")) {
			PdfPTable pptAdresseDroite = new PdfPTable(1);
			pptAdresseDroite.setWidthPercentage(100);
			
			cell.setPhrase(new Phrase("Contact : "+id.interlocuteur, fontAdrContact));
			pptAdresseDroite.addCell(cell);
			if (!id.interMail.equals("")) {
				cell.setPhrase(new Phrase("E-Mail : "+id.interMail, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interTel.equals("")) {
				cell.setPhrase(new Phrase("T\u00E9l : "+id.interTel, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interFax.equals("")) {
				cell.setPhrase(new Phrase("Fax : "+id.interFax, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			PdfPCell cellPartieDroite = new PdfPCell(pptAdresseDroite);
			pptAdresses.addCell(cellPartieDroite);
		}
		
		return pptAdresses;
	}
	
	
	private PdfPTable makeAdrLiv() throws DocumentException {
		
		PdfPTable pptAdresses = new PdfPTable(id.interlocuteurLiv.equals("")?1:2);
		pptAdresses.setWidthPercentage(100);
		if (!id.interlocuteurLiv.equals("")) {
			int[] widths = {50,50};
			pptAdresses.setWidths(widths);
		}
		
		PdfPCell cell = new PdfPCell(pptAdresses.getDefaultCell());
		if (!id.interlocuteurLiv.equals("")) { cell.setColspan(2); }
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);
		cell.setPhrase(new Phrase("LIVRAISON", fontSmallTitre));
		pptAdresses.addCell(cell);
		
		PdfPTable pptAdresseGauche = new PdfPTable(1);
		pptAdresseGauche.setWidthPercentage(100);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(Color.WHITE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.denominationLiv, fontAdresse));
		pptAdresseGauche.addCell(cell);
		cell.setPhrase(new Phrase(id.adresse1Liv, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.adresse2Liv.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Liv, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		if (!id.adresse3Liv.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Liv, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		cell.setPhrase(new Phrase(id.codePostalLiv+" "+id.villeLiv, fontAdresse));
		pptAdresseGauche.addCell(cell);
		if (!id.paysLiv.equals("")) {
			cell.setPhrase(new Phrase(id.paysLiv, fontAdresse));
			pptAdresseGauche.addCell(cell);
		}
		
		PdfPCell cellPartieGauche = new PdfPCell(pptAdresseGauche);
		pptAdresses.addCell(cellPartieGauche);
		
		if (!id.interlocuteurLiv.equals("")) {
			PdfPTable pptAdresseDroite = new PdfPTable(1);
			pptAdresseDroite.setWidthPercentage(100);
			
			cell.setPhrase(new Phrase("Contact : "+id.interlocuteurLiv, fontAdrContact));
			pptAdresseDroite.addCell(cell);
			if (!id.interMailLiv.equals("")) {
				cell.setPhrase(new Phrase("E-Mail : "+id.interMailLiv, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interTelLiv.equals("")) {
				cell.setPhrase(new Phrase("T\u00E9l : "+id.interTelLiv, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			if (!id.interFaxLiv.equals("")) {
				cell.setPhrase(new Phrase("Fax : "+id.interFaxLiv, fontAdrContact));
				pptAdresseDroite.addCell(cell);
			}
			PdfPCell cellPartieDroite = new PdfPCell(pptAdresseDroite);
			pptAdresses.addCell(cellPartieDroite);
		}
		
		return pptAdresses;
	}
	
	
	private PdfPTable makeAdrEnvoi() throws DocumentException {

		PdfPTable pptInfosFacture = new PdfPTable(2);

		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());

		cell.setColspan(2);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setMinimumHeight(0);
		cell.setPaddingLeft(0);
		cell.setPaddingBottom(0);
		
		cell.setBorder(PdfPCell.NO_BORDER);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(id.denominationEnvoi, fontAdrGras));
		pptInfosFacture.addCell(cell);
		
		cell.setPhrase(new Phrase(id.adresse1Envoi, fontAdrGras));
		pptInfosFacture.addCell(cell);
		
		if (!id.adresse2Envoi.equals("")) {
			cell.setPhrase(new Phrase(id.adresse2Envoi, fontAdrGras));
			pptInfosFacture.addCell(cell);
		}
		
		if (!id.adresse3Envoi.equals("")) {
			cell.setPhrase(new Phrase(id.adresse3Envoi, fontAdrGras));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setPhrase(new Phrase(id.codePostalEnvoi+" "+id.villeEnvoi, fontAdrGras));
		pptInfosFacture.addCell(cell);
		
		if (!id.paysEnvoi.equals("")) {
			cell.setPhrase(new Phrase(id.paysEnvoi, fontAdrGras));
			pptInfosFacture.addCell(cell);
		}
		
		cell.setPhrase(new Phrase("", fontLine));
		pptInfosFacture.addCell(cell);

		PdfPTable pptGlobale = new PdfPTable(2);
		int[] widths = {85,15};
		pptGlobale.setWidths(widths);

		PdfPCell cellGlob = new PdfPCell(pptInfosFacture);
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		cellGlob = new PdfPCell(pptGlobale.getDefaultCell());
		cellGlob.setPhrase(new Phrase(""));
		cellGlob.setBorder(PdfPCell.NO_BORDER);

		pptGlobale.addCell(cellGlob);

		return pptGlobale;
	}


	private PdfPTable makeInfosFacture() throws DocumentException {

		PdfPTable pptInfosFacture = new PdfPTable(2);

		int[] widths = {60,40};
		pptInfosFacture.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptInfosFacture.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingLeft(4);
		cell.setPaddingBottom(4);

		cell.setPhrase(new Phrase("Facture N\u00B0 "+ id.numeroFacture, fontInfosFact));
		pptInfosFacture.addCell(cell);

		cell.setPhrase(new Phrase("Date : "+ formatDate.format(new Date(id.dateFacture)), fontInfosFact));
		pptInfosFacture.addCell(cell);

		String codeClient = (id.clientId.length()>0?"Client N\u00B0 "+ id.clientId +" - ":"Client : ") + id.raisonSocialeDest;

		cell.setColspan(2);
		cell.setPhrase(new Phrase(codeClient, fontInfosFact));
		pptInfosFacture.addCell(cell);

		if (id.interlocuteur.length()>0) {
			cell.setPhrase(new Phrase("Interlocuteur : "+ id.interlocuteur, fontInfosFact));
			pptInfosFacture.addCell(cell);
		}

		cell.setBorder(PdfPCell.NO_BORDER);

		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(5);
		pptInfosFacture.addCell(cell);
		cell.setMinimumHeight(0);

		cell.setPhrase(new Phrase("Page "+ currentPage +" / "+ totalPages, fontInfosFact));
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		pptInfosFacture.addCell(cell);

		cell.setPhrase(new Phrase("", fontLine));
		cell.setFixedHeight(55);
		pptInfosFacture.addCell(cell);

		return pptInfosFacture;		
	}


	private PdfPTable makeTitre() {

		PdfPTable pptTitre = new PdfPTable(1);
		pptTitre.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptTitre.getDefaultCell());		
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setBackgroundColor(new Color(175,175,175));
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setFixedHeight(30);
		cell.setPaddingBottom(5);
		cell.setPhrase(new Phrase("F     A     C     T     U     R     E", fontTitre));

		pptTitre.addCell(cell);

		return pptTitre;	
	}


	private PdfPTable makeResponsable() {

		PdfPTable pptResponsable = new PdfPTable(1);
		pptResponsable.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptResponsable.getDefaultCell());		

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setFixedHeight(20);
		cell.setPaddingLeft(5);

		if (id.directe==1 && id.codeFournisseur.length()>0) {
			cell.setPhrase(new Phrase("Num\u00E9ro Fournisseur : "+ id.codeFournisseur, fontCadre1));
			pptResponsable.addCell(cell);
		}
		else if (id.codeFournisseur.length()>0 || !id.refAffaire.equals("") || !id.refCommande.equals("") || !id.refBL.equals("")) {
			cell.setBorder(PdfPCell.BOX-PdfPCell.BOTTOM);
			cell.setMinimumHeight(0);
			cell.setPhrase(new Phrase("", fontCadre1));
			pptResponsable.addCell(cell);
			cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
			cell.setFixedHeight(20);
			if (id.codeFournisseur.length()>0) {
				cell.setPhrase(new Phrase("Num\u00E9ro Fournisseur : "+ id.codeFournisseur, fontCadre1));
				pptResponsable.addCell(cell);
				cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
			}

			cell.setMinimumHeight(0);
			if (!id.refAffaire.equals("")) {
				cell.setPhrase(new Phrase("R\u00E9f. Affaire : "+ id.refAffaire, fontLegale));
				pptResponsable.addCell(cell);
			}
			
			if (!id.numCommande.equals("")) {
				cell.setPhrase(new Phrase("N\u00B0 Commande : "+ id.numCommande, fontLegale));
				pptResponsable.addCell(cell);
			}

			if (!id.refCommande.equals("")) {
				cell.setPhrase(new Phrase("R\u00E9f. Commande : "+ id.refCommande, fontLegale));
				pptResponsable.addCell(cell);
			}

			if (!id.refBL.equals("")) {
				cell.setPhrase(new Phrase("R\u00E9f. BL : "+ id.refBL, fontLegale));
				pptResponsable.addCell(cell);
			}
			cell.setBorder(PdfPCell.BOX-PdfPCell.TOP);
			cell.setPhrase(new Phrase("", fontCadre1));
			pptResponsable.addCell(cell);
		}
		return pptResponsable;
	}


	private PdfPTable makeEntetesColonnes() throws DocumentException {

		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());		

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingBottom(4);
		cell.setMinimumHeight(40);

		cell.setPhrase(new Phrase("R\u00E9f\u00E9rence", fontEnteteColonne));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("D\u00E9signation", fontEnteteColonne));
		pptCols.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase("Coul.", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		if (tail) {
			cell.setPhrase(new Phrase("Taille", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		
		if (numLot) {
			cell.setPhrase(new Phrase("N\u00B0 Lot", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		if (nbPieces) {
			cell.setPhrase(new Phrase("Nb Pc", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		if (datePeremption) {
			cell.setPhrase(new Phrase("DLC", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		cell.setPhrase(new Phrase("Quantit\u00E9", fontEnteteColonne));
		pptCols.addCell(cell);

		if (unite) {
			cell.setPhrase(new Phrase("Unit\u00E9", fontEnteteColonne));
			pptCols.addCell(cell);
		}

		cell.setPhrase(new Phrase((id.editionTTC?"PU TTC":"PU HT"), fontEnteteColonne));
		pptCols.addCell(cell);
		if (rist) {
			cell.setPhrase(new Phrase("Rist. %", fontEnteteColonne));
			pptCols.addCell(cell);
		}
		cell.setPhrase(new Phrase((id.editionTTC?"Montant TTC":"Montant HT"), fontEnteteColonne));		
		pptCols.addCell(cell);

		return pptCols;
	}	


	private PdfPTable makeLineArticle(LigneFacture lf) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);

		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase(lf.reference, fontLine));
		pptLine.addCell(cell);		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase(lf.designation, fontLine));
		pptLine.addCell(cell);

		if (coul) {
			cell.setPhrase(new Phrase(lf.couleur, fontLine));
			pptLine.addCell(cell);
		}
		if (tail) {
			cell.setPhrase(new Phrase(lf.taille, fontLine));
			pptLine.addCell(cell);
		}
		
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		
		if (numLot) {
			cell.setPhrase(new Phrase(lf.num_lot, fontLine));
			pptLine.addCell(cell);
		}
		if (nbPieces) {
			if (lf.nb_pieces != 0)
				cell.setPhrase(new Phrase(""+numFormatCap.format(lf.nb_pieces), fontLine));
			else
				cell.setPhrase(new Phrase("", fontLine));			
			pptLine.addCell(cell);
		}
		if (datePeremption) {
			if (lf.date_peremption != 0)
				cell.setPhrase(new Phrase(""+ formatDate.format(new Date(lf.date_peremption)), fontLine));
			else
				cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
		}

		cell.setPhrase(new Phrase(""+ numFormatCap.format(lf.quantite), fontLine));
		pptLine.addCell(cell);
		
		if (unite) {
			cell.setPhrase(new Phrase(lf.unite, fontLine));
			pptLine.addCell(cell);
		}

		cell.setPaddingRight(3);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase(numFormat4.format(lf.prixUnitaire), fontLine));
		pptLine.addCell(cell);
		if (rist) {
			if (lf.ristourne!=0) {
				cell.setPhrase(new Phrase(numFormat.format(lf.ristourne), fontLine));
			}
			else {
				cell.setPhrase(new Phrase("", fontLine));
			}
			pptLine.addCell(cell);
		}

		cell.setPhrase(new Phrase(numFormat.format(lf.montant), fontLine));		
		pptLine.addCell(cell);		


		if (lf.refBL.length()>0) {
			cell.setPaddingLeft(5);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPaddingRight(0);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lf.refBL, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
		}

		if (lf.detail_1.length()>0) {
			cell.setPaddingLeft(5);
			cell.setPaddingRight(0);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lf.detail_1, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);		
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
		}

		if (lf.detail_2.length()>0) {
			cell.setPaddingLeft(5);
			cell.setPaddingRight(0);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase(lf.detail_2, fontLine));
			pptLine.addCell(cell);
			cell.setPhrase(new Phrase("", fontLine));
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			pptLine.addCell(cell);
			if (rist) pptLine.addCell(cell);
			if (coul) pptLine.addCell(cell);		
			if (tail) pptLine.addCell(cell);
			if (unite) pptLine.addCell(cell);
			if (numLot) pptLine.addCell(cell);
			if (nbPieces) pptLine.addCell(cell);
			if (datePeremption) pptLine.addCell(cell);
		}

		return pptLine;	
	}


	private PdfPTable makeLineNomenclature(LigneFacture lf) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);


		PdfPTable pptArtNom = new PdfPTable(unite?3:2);
		pptArtNom.setWidthPercentage(100);
		int[] widths = {10,90};
		int[] widthsUnite = {10,10,80};
		pptArtNom.setWidths(unite?widthsUnite:widths);

		PdfPCell cellArtNom = new PdfPCell(pptArtNom.getDefaultCell());
		cellArtNom.setBorder(PdfPCell.NO_BORDER);

		cellArtNom.setPhrase(new Phrase(numFormatCap.format(lf.quantite), fontArtNom));
		pptArtNom.addCell(cellArtNom);
		
		if (unite) {
			cellArtNom.setPhrase(new Phrase(lf.unite, fontArtNom));
			pptArtNom.addCell(cellArtNom);
		}

		cellArtNom.setPhrase(new Phrase(lf.designation, fontArtNom));
		pptArtNom.addCell(cellArtNom);

		PdfPCell cellDesc = new PdfPCell(pptArtNom);
		cellDesc.setBorder(PdfPCell.NO_BORDER);
		cellDesc.setPaddingLeft(10);
		pptLine.addCell(cellDesc);

		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);

		if (rist) pptLine.addCell(cell);		
		if (coul) pptLine.addCell(cell);		
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);

		return pptLine;	
	}


	private PdfPTable makeLineCommentaire(LigneFacture lf) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		

		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		cell.setPhrase(new Phrase(lf.designation, fontLine));
		pptLine.addCell(cell);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);
		pptLine.addCell(cell);
		pptLine.addCell(cell);

		if (rist) pptLine.addCell(cell);
		if (coul) pptLine.addCell(cell);		
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);
		
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);

		return pptLine;	
	}


	private PdfPTable makeSousTotal(double repriseSousTotal) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());		

		cell.setBorder(PdfPCell.RIGHT+PdfPCell.LEFT+PdfPCell.BOTTOM);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setPaddingBottom(6);
		cell.setPaddingTop(3);

		cell.setPaddingLeft(5);
		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPaddingRight(3);
		cell.setPhrase(new Phrase((id.editionTTC?"Sous-total TTC":"Sous-total HT"), fontTotaux));
		pptLine.addCell(cell);
		cell.setPaddingLeft(0);	

		cell.setPhrase(new Phrase("", fontLine));
		pptLine.addCell(cell);	

		cell.setPhrase(new Phrase("", fontTotaux));
		pptLine.addCell(cell);
		
		if (rist) pptLine.addCell(cell);
		if (coul) pptLine.addCell(cell);
		if (tail) pptLine.addCell(cell);
		if (unite) pptLine.addCell(cell);		
		if (numLot) pptLine.addCell(cell);
		if (nbPieces) pptLine.addCell(cell);
		if (datePeremption) pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(numFormat.format(repriseSousTotal), fontTotaux));		
		pptLine.addCell(cell);

		return pptLine;	
	}


	private PdfPTable makeTotauxLast() throws DocumentException {

		PdfPTable pptTotaux = new PdfPTable(3);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {50,15,35};
		pptTotaux.setWidths(widths);		

		PdfPCell cell = new PdfPCell(makeCadreTvaDivers());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);		
		pptTotaux.addCell(cell);

		cell = new PdfPCell(pptTotaux.getDefaultCell());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);		
		cell.setPhrase(new Phrase(""));
		pptTotaux.addCell(cell);

		cell = new PdfPCell(makeCadreTotaux());
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);		
		pptTotaux.addCell(cell);		

		return pptTotaux;	
	}	


	private PdfPTable makeTotauxMiddle() throws DocumentException {

		PdfPTable pptTotaux = new PdfPTable(3);
		pptTotaux.setWidthPercentage(100);
		int[] widths = {50,15,35};
		pptTotaux.setWidths(widths);		

		PdfPCell cell = new PdfPCell(pptTotaux.getDefaultCell());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.LEFT);
		cell.setPaddingBottom(3);	
		pptTotaux.addCell(cell);

		cell = new PdfPCell(pptTotaux.getDefaultCell());	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP);		
		cell.setPhrase(new Phrase(""));
		pptTotaux.addCell(cell);

		cell = new PdfPCell(makeCadreSousTotal());
		cell.setVerticalAlignment(PdfPCell.ALIGN_TOP);	
		cell.setBorder(PdfPCell.BOTTOM+PdfPCell.TOP+PdfPCell.RIGHT);		
		pptTotaux.addCell(cell);			

		return pptTotaux;	
	}	


	private PdfPTable makeCadreTVA() {

		PdfPTable pptTva = new PdfPTable(3);
		pptTva.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptTva.getDefaultCell());

		cell.setBorder(PdfPCell.BOX);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPhrase(new Phrase("Base HT", fontEnteteColonne));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Taux TVA", fontEnteteColonne));
		pptTva.addCell(cell);
		cell.setPhrase(new Phrase("Montant TVA", fontEnteteColonne));	
		pptTva.addCell(cell);

		Iterator<Integer> li = id.ventilTVA.keySet().iterator();
		
		while (li.hasNext()) {
			CalculVentilTVA vtva = id.ventilTVA.get(li.next());

			cell.setPaddingRight(3);
			cell.setBorder(PdfPCell.BOX-PdfPCell.TOP-PdfPCell.BOTTOM);
			cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			cell.setPhrase(new Phrase(numFormat.format(vtva.getMontantHT()), fontTotaux));		
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(vtva.getTauxTVA()) +" %"+ (vtva.getNPR()?" NPR":""), fontTotaux));
			pptTva.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(vtva.getMontantTVA()), fontTotaux));
			pptTva.addCell(cell);
		}		

		cell.setBorder(PdfPCell.TOP);
		cell.setColspan(3);
		cell.setPhrase(new Phrase(""));
		cell.setFixedHeight(0);
		pptTva.addCell(cell);

		return pptTva;
	}


	private PdfPTable makeCadreDivers() throws DocumentException {

		PdfPTable pptDivers = new PdfPTable(2);
		pptDivers.setWidthPercentage(100);
		int[] widths = {30,70};
		pptDivers.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptDivers.getDefaultCell());

		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setColspan(2);

		// formule pour exonration car Assujetti
	    if (id.assujetti && id.numTvaClient!="" && id.zoneUE && !id.codePaysLiv.equals("FR") && !id.regimeTVA.equals("T") && !id.typePays.equals("D")) {
	      cell.setPhrase(new Phrase("Exonration de TVA art.262 ter, I du CGI", fontModalites));// pro, num_tva, UE
	      pptDivers.addCell(cell);
	    }
	    
	    // formule pour exonration  l'International
	    if (!id.zoneUE && !id.regimeTVA.equals("T")) {
	      cell.setPhrase(new Phrase("Exonration de TVA en application de l'art.262 I du CGI", fontModalites)); //International
	      pptDivers.addCell(cell);
	    }
	    
	    // formule pour exonration DOM-TOM
	    if (id.typePays.equals("D") && !id.regimeTVA.equals("T") && (id.totalTVA==0)) {
	      cell.setPhrase(new Phrase("Exonration de TVA en application de l'art.264-2 du CGI", fontModalites));//dom
	      pptDivers.addCell(cell);
	    }
	    
		// formule pour application de TVA dans le pays de livraison
	    if (id.TVALiv && id.regimeTVA.equals("G") && (id.totalTVA!=0)) {
	      cell.setPhrase(new Phrase("Application de la TVA du pays de livraison en vertu de l'article 258 A du CGI", fontModalites));//taxe_arrivee
	      pptDivers.addCell(cell);
	    }
	    
	    int nbAcomptes = id.lignesAcomptes.size();
	    if (nbAcomptes>0) {
	    	PdfPTable pptAcomptes = new PdfPTable(3);
	    	pptAcomptes.setWidthPercentage(100);
			int[] widthsAcpte = {33,33,34};
			pptAcomptes.setWidths(widthsAcpte);
			
	    	PdfPCell cellAcompte = new PdfPCell(pptAcomptes.getDefaultCell());
	    	cellAcompte.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
	    	cellAcompte.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
	    	cellAcompte.setPhrase(new Phrase("Date", fontEnteteColonne));
			pptAcomptes.addCell(cellAcompte);
			cellAcompte.setPhrase(new Phrase("Montant", fontEnteteColonne));
			pptAcomptes.addCell(cellAcompte);
			cellAcompte.setPhrase(new Phrase("N\u00B0 Facture acompte", fontEnteteColonne));
			pptAcomptes.addCell(cellAcompte);
			
			cellAcompte.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
			for (int i=0; i<nbAcomptes; i++) {
				Ligne ligne = id.lignesAcomptes.get(i);
				cellAcompte.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
				cellAcompte.setPhrase(new Phrase(formatDate.format(new Date(ligne.date)), fontTotaux));
				pptAcomptes.addCell(cellAcompte);
				cellAcompte.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
				cellAcompte.setPhrase(new Phrase(numFormat.format(ligne.montant), fontTotaux));
				pptAcomptes.addCell(cellAcompte);
				cellAcompte.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
				cellAcompte.setPhrase(new Phrase(ligne.libelle, fontTotaux));
				pptAcomptes.addCell(cellAcompte);
			}
			cellAcompte.setPhrase(new Phrase("", fontTotaux));
			cellAcompte.setFixedHeight(1);
			cellAcompte.setBorder(PdfPCell.BOX-PdfPCell.TOP);
			pptAcomptes.addCell(cellAcompte);
			pptAcomptes.addCell(cellAcompte);
			pptAcomptes.addCell(cellAcompte);
			
			cell.setPhrase(new Phrase("Acomptes :", fontModalites));
			pptDivers.addCell(cell);
			
			PdfPCell cellTabAcomptes = new PdfPCell(pptAcomptes);
			cellTabAcomptes.setBorder(PdfPCell.NO_BORDER);
			cellTabAcomptes.setColspan(2);
			pptDivers.addCell(cellTabAcomptes);
			
			cell.setPhrase(new Phrase("", fontLine));
			cell.setFixedHeight(10);
			pptDivers.addCell(cell);
			cell.setMinimumHeight(0);
	    }
	    
    	int nbEcheances = id.lignesEcheances.size();
    	if (nbEcheances==1) {
			Ligne ligne = id.lignesEcheances.get(0);
	    
			if (ligne.date!=0) {
				cell.setPhrase(new Phrase("Date d'\u00E9ch\u00E9ance : "+ formatDate.format(new Date(ligne.date)), fontModalites));
				pptDivers.addCell(cell);
			}
	
			cell.setPhrase(new Phrase("Mode de r\u00E8glement : "+ ligne.libelle, fontModalites));
			pptDivers.addCell(cell);
	    } else if (nbEcheances>0) {
	    	PdfPTable pptEcheances = new PdfPTable(3);
	    	pptEcheances.setWidthPercentage(100);
			int[] widthsEch = {33,33,34};
			pptEcheances.setWidths(widthsEch);

			PdfPCell cellEch = new PdfPCell(pptEcheances.getDefaultCell());
			cellEch.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
			cellEch.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
			cellEch.setPhrase(new Phrase("Date", fontEnteteColonne));
			pptEcheances.addCell(cellEch);
			cellEch.setPhrase(new Phrase("Montant", fontEnteteColonne));
			pptEcheances.addCell(cellEch);
			cellEch.setPhrase(new Phrase("Mode de r\u00E8glement", fontEnteteColonne));
			pptEcheances.addCell(cellEch);
			
			cellEch.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
			for (int i=0; i<nbEcheances; i++) {
				Ligne ligne = id.lignesEcheances.get(i);
				cellEch.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
				cellEch.setPhrase(new Phrase(formatDate.format(new Date(ligne.date)), fontTotaux));
				pptEcheances.addCell(cellEch);
				cellEch.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
				cellEch.setPhrase(new Phrase(numFormat.format(ligne.montant), fontTotaux));
				pptEcheances.addCell(cellEch);
				cellEch.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
				cellEch.setPhrase(new Phrase(ligne.libelle, fontTotaux));
				pptEcheances.addCell(cellEch);
			}
			cellEch.setPhrase(new Phrase("", fontTotaux));
			cellEch.setFixedHeight(1);
			cellEch.setBorder(PdfPCell.BOX-PdfPCell.TOP);
			pptEcheances.addCell(cellEch);
			pptEcheances.addCell(cellEch);
			pptEcheances.addCell(cellEch);
			
			cell.setPhrase(new Phrase("Ech\u00E9ances :", fontModalites));
			pptDivers.addCell(cell);
			
			PdfPCell cellTabEch = new PdfPCell(pptEcheances);
			cellTabEch.setBorder(PdfPCell.NO_BORDER);
			cellTabEch.setColspan(2);
			pptDivers.addCell(cellTabEch);
			
			cell.setPhrase(new Phrase("", fontLine));
			cell.setFixedHeight(10);
			pptDivers.addCell(cell);
			cell.setMinimumHeight(0);
	    }

		if (!id.numTvaClient.equals("")) {
			cell.setPhrase(new Phrase("", fontModalites));
			pptDivers.addCell(cell);
			cell.setPhrase(new Phrase("N\u00B0 TVA Client : "+ id.numTvaClient, fontModalites));
			pptDivers.addCell(cell);
		}

		if (id.ecoTaxe>0) {
			cell.setPhrase(new Phrase("", fontModalites));
			pptDivers.addCell(cell);
			cell.setPhrase(new Phrase("Montant eco-participation : "+ numFormat.format(id.ecoTaxe) +" \u20AC HT", fontWeb));
			pptDivers.addCell(cell);
		}

		cell.setMinimumHeight(10);
		cell.setPhrase(new Phrase("", fontModalites));
		pptDivers.addCell(cell);		

		return pptDivers;
	}


	private PdfPTable makeCadreTvaDivers() throws DocumentException {
		PdfPTable pptTvaDivers = new PdfPTable(1);
		pptTvaDivers.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(makeCadreTVA());	
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(3);
		cell.setPaddingTop(3);
		pptTvaDivers.addCell(cell);		

		cell = new PdfPCell(makeCadreDivers());
		cell.setPaddingLeft(3);
		cell.setPaddingBottom(3);	
		cell.setBorder(PdfPCell.NO_BORDER);		
		pptTvaDivers.addCell(cell);

		return pptTvaDivers;
	}	


	private PdfPTable makeCadreSousTotal() throws DocumentException {

		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		cell.setBorder(PdfPCell.NO_BORDER);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPhrase(new Phrase((id.editionTTC?"Sous-total TTC :":"Sous-total HT :"), fontTotaux));
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(sousTotalCourant), fontTotaux));
		pptCadreTotaux.addCell(cell);

		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);

		return pptCadreTotaux;	
	}


	private PdfPTable makeCadreTotaux() throws DocumentException {

		PdfPTable pptCadreTotaux = new PdfPTable(2);
		pptCadreTotaux.setWidthPercentage(100);
		int[] widths = {55,45};
		pptCadreTotaux.setWidths(widths);

		PdfPCell cell = new PdfPCell(pptCadreTotaux.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setBorder(PdfPCell.NO_BORDER);		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);

		if (id.tRemise>0 || (!id.sansarticles && id.fraisPort>0)) {
			cell.setPhrase(new Phrase(id.exonerationTVA?"Montant :":(id.editionTTC?"Montant TTC :":"Montant HT :"), fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.montantBase) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);

			if (id.tRemise>0) {		
				cell.setPhrase(new Phrase("Remise ("+ numFormatCap.format(id.tRemise) +" %) :", fontTotaux));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(""+ numFormat.format(id.mRemise) +" \u20AC", fontTotaux));
				pptCadreTotaux.addCell(cell);
			}

			if (!id.sansarticles && id.fraisPort>0) {
				cell.setPhrase(new Phrase("Frais de livraison :", fontTotaux));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(numFormat.format(id.fraisPort) +" \u20AC", fontTotaux));
				pptCadreTotaux.addCell(cell);
				
				if (id.tRemiseFP>0) {		
					cell.setPhrase(new Phrase("Remise sur frais de port ("+ numFormatCap.format(id.tRemiseFP) +" %) :", fontTotaux));
					pptCadreTotaux.addCell(cell);
					cell.setPhrase(new Phrase(""+ numFormat.format(id.mRemiseFP) +" \u20AC", fontTotaux));
					pptCadreTotaux.addCell(cell);
				}
			}
		}

		cell.setPhrase(new Phrase(id.exonerationTVA?"Total :":(id.editionTTC?"Total TTC :":"Total HT :"),id.editionTTC || id.exonerationTVA?fontTTC:fontTotaux));
		pptCadreTotaux.addCell(cell);
		cell.setPhrase(new Phrase(numFormat.format(id.totalBase) +" \u20AC",id.editionTTC || id.exonerationTVA?fontTTC:fontTotaux));
		pptCadreTotaux.addCell(cell);

		if (!id.exonerationTVA) {
			cell.setPhrase(new Phrase((id.editionTTC?"Dont TVA :":"Total TVA :"), fontTotaux));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.totalTVA) +" \u20AC", fontTotaux));
			pptCadreTotaux.addCell(cell);
			
			if (!id.editionTTC) {
				cell.setPhrase(new Phrase("Total TTC :", fontTTC));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(numFormat.format(id.montantTTC) +" \u20AC", fontTTC));
				pptCadreTotaux.addCell(cell);		
			}
		}

		if (id.tEscompte>0 || id.acompte>0) {
			cell.setPhrase(new Phrase("", fontTotaux));
			cell.setColspan(2);
			cell.setFixedHeight(10);
			pptCadreTotaux.addCell(cell);
			cell.setColspan(1);
			cell.setMinimumHeight(0);
			
			if (id.tEscompte>0) {
				cell.setPhrase(new Phrase("Escompte ("+ numFormatCap.format(id.tEscompte) +" %) :", fontTotaux));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(numFormat.format(id.mEscompte) +" \u20AC", fontTotaux));
				pptCadreTotaux.addCell(cell);
			}
			
			if (id.acompte>0) {
				cell.setPhrase(new Phrase("Acompte :", fontTotaux));
				pptCadreTotaux.addCell(cell);
				cell.setPhrase(new Phrase(numFormat.format(id.acompte) +" \u20AC", fontTotaux));
				pptCadreTotaux.addCell(cell);
			}

			cell.setPhrase(new Phrase("Net \u00E0 payer :", fontTTC));
			pptCadreTotaux.addCell(cell);
			cell.setPhrase(new Phrase(numFormat.format(id.netAPayer) +" \u20AC", fontTTC));
			pptCadreTotaux.addCell(cell);	
		}
		cell.setColspan(2);
		cell.setPhrase(new Phrase(""));
		pptCadreTotaux.addCell(cell);

		return pptCadreTotaux;
	}


	private PdfPTable makeZoneCommentaire() throws DocumentException {		

		PdfPTable pptCom = new PdfPTable(nbCols);
		pptCom.setWidthPercentage(100);
		pptCom.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptCom.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT + PdfPCell.RIGHT);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);		
		cell.setPaddingLeft(5);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cell.setPhrase(new Phrase("", fontLine));
		pptCom.addCell(cell);
		cell.setPhrase(new Phrase(id.comFin + id.mentions, fontLine));
		pptCom.addCell(cell);
		cell.setPhrase(new Phrase("", fontLine));
		pptCom.addCell(cell);
		pptCom.addCell(cell);
		pptCom.addCell(cell);

		if (rist) pptCom.addCell(cell);
		if (coul) pptCom.addCell(cell);		
		if (tail) pptCom.addCell(cell);
		if (unite) pptCom.addCell(cell);	
		
		if (numLot) pptCom.addCell(cell);
		if (nbPieces) pptCom.addCell(cell);
		if (datePeremption) pptCom.addCell(cell);

		return pptCom;		
	}	


	private PdfPTable makeCadreBas2() {

		PdfPTable pptCadre = new PdfPTable(1);
		pptCadre.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptCadre.getDefaultCell());

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

		Phrase reserve = new Phrase();
		Chunk ch1 = new Chunk("RESERVE DE PROPRIETE : ", fontCadre1PetitGras);
		Chunk ch2 = new Chunk("loi du 12/05/1980 N\u00B0 80335 - Le mat\u00E9riel livr\u00E9 reste l'enti\u00E8re propri\u00E9t\u00E9 de "+ id.raisonSociale +" jusqu'au paiement int\u00E9gral du prix fix\u00E9 ci-dessus.", fontCadre1Petit);
		Chunk ch3 = new Chunk("\nInt\u00E9r\u00EAt de retard : ", fontCadre1PetitGras);
		Chunk ch4 = new Chunk("Tout retard de paiement fera l'objet de l'application d'un int\u00E9r\u00EAt de retard \u00E9gal \u00E0 3 fois le taux d'int\u00E9r\u00EAt l\u00E9gal (Loi n\u00B02008-776 du 4 Ao\u00FBt 2008)", fontCadre1Petit);

		reserve.add(ch1);
		reserve.add(ch2);
		reserve.add(ch3);
		reserve.add(ch4);

		cell.setPhrase(reserve);
		cell.setPaddingBottom(4);

		pptCadre.addCell(cell);

		return pptCadre;
	}
	
	
	private PdfPTable makeLCR() throws DocumentException {
		PdfPTable pptCadre = new PdfPTable(1);
		pptCadre.setWidthPercentage(100);
		
		PdfPTable pptPointilles = new PdfPTable(100);
		pptPointilles.setWidthPercentage(100);
		int[] widthsPointilles = new int[100];
		for (int i=0; i<100; i++) {
			widthsPointilles[i] = 1;
		}
		pptPointilles.setWidths(widthsPointilles);
		
		PdfPCell cellPointilles = new PdfPCell(pptPointilles.getDefaultCell());
		cellPointilles.setPhrase(new Phrase("", fontAdrLCR));	
		cellPointilles.setBorder(PdfPCell.NO_BORDER);
		for (int i=0; i<100; i++) {
			if (i%2==0) {
				cellPointilles.setBorder(PdfPCell.BOTTOM);
				cellPointilles.setBorderWidth(1.2f);
			}
			else { cellPointilles.setBorder(PdfPCell.NO_BORDER); }
			pptPointilles.addCell(cellPointilles);
		}
		
		PdfPCell cell = new PdfPCell(pptPointilles);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);
		cell = new PdfPCell(pptCadre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setPhrase(new Phrase("", fontAdrLCR));
		cell.setFixedHeight(5);
		pptCadre.addCell(cell);
		
		PdfPTable pptLigne1 = new PdfPTable(4);
		pptLigne1.setWidthPercentage(100);
		int[] widthsLigne1 = {30,25,35,10};
		pptLigne1.setWidths(widthsLigne1);
		PdfPTable pptAdresseLCR = new PdfPTable(1);
		pptAdresseLCR.setWidthPercentage(100);
		PdfPCell cellLCR = new PdfPCell(pptAdresseLCR.getDefaultCell());
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);		
		cellLCR.setPhrase(new Phrase(id.raisonSociale, fontAdrGrasLCR));		
		pptAdresseLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase(id.adresse1, fontAdrLCR));		
		pptAdresseLCR.addCell(cellLCR);
		if (!id.adresse2.equals("")) {
			cellLCR.setPhrase(new Phrase(id.adresse2, fontAdrLCR));
			pptAdresseLCR.addCell(cellLCR);
		}
		if (!id.adresse3.equals("")) {
			cellLCR.setPhrase(new Phrase(id.adresse3, fontAdrLCR));
			pptAdresseLCR.addCell(cellLCR);
		}
		cellLCR.setPhrase(new Phrase(id.codePostal+" "+id.ville, fontAdrLCR));		
		pptAdresseLCR.addCell(cellLCR);
		String telfax = "";
		if (id.tel.length()>0) {
			telfax = "T\u00E9l: "+ id.tel;
			if (id.fax.length()>0) {
				telfax += " - Fax: "+ id.fax;
			}
		}
		else if (id.fax.length()>0) {
			telfax = "Fax: "+ id.fax;
		}
		if (telfax.length()>0) {
			cellLCR.setPhrase(new Phrase(telfax, fontAdrLCR));		
			pptAdresseLCR.addCell(cellLCR);
		}
		cellLCR.setPhrase(new Phrase("", fontAdrLCR));		
		pptAdresseLCR.addCell(cellLCR);
		PdfPCell cellLigne = new PdfPCell(pptAdresseLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne1.addCell(cellLigne);		
		pptAdresseLCR = new PdfPTable(1);
		pptAdresseLCR.setWidthPercentage(100);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellLCR.setPhrase(new Phrase("Contre cette LETTRE DE CHANGE", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("stipul\u00E9e SANS FRAIS", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("veuillez payer la somme indiqu\u00E9e", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("ci-dessous \u00E0 l'ordre de", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLigne = new PdfPCell(pptAdresseLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setVerticalAlignment(PdfPCell.ALIGN_BOTTOM);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne1.addCell(cellLigne);
		cellLigne = new PdfPCell(pptLigne1.getDefaultCell());
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontAdrLCR));
		pptLigne1.addCell(cellLigne);
		pptAdresseLCR = new PdfPTable(1);
		pptAdresseLCR.setWidthPercentage(100);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellLCR.setPhrase(new Phrase("mention", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("L.C.R.", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("s'il y a lieu", fontAdrLCR));
		pptAdresseLCR.addCell(cellLCR);
		cellLigne = new PdfPCell(pptAdresseLCR);
		cellLigne.setBorder(PdfPCell.LEFT);
		cellLigne.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne1.addCell(cellLigne);
		cellLigne = new PdfPCell(pptLigne1.getDefaultCell());
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);		
		String lieuDate = "\u00C0 ";
		for (int i=0; i<35; i++) {
			lieuDate += "_";
		}
		lieuDate += " LE __";		
		Chunk c1 = new Chunk(lieuDate, fontLineLCR);
		Chunk c2;
		try {
			Image fleche = Image.getInstance(id.urlPetiteFlecheLCR);
			fleche.scaleAbsolute(3, 5);
			c2 = new Chunk(fleche, -2, -6);
		} catch (IOException e) {
			c2 = new Chunk("");
		}
		Phrase p1 = new Phrase();
		p1.add(c1); p1.add(c2);
		cellLigne.setPhrase(p1);
		pptLigne1.addCell(cellLigne);		
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		cellLigne.setColspan(3);
		pptLigne1.addCell(cellLigne);		
		cell = new PdfPCell(pptLigne1);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);
		cell = new PdfPCell(pptCadre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setPhrase(new Phrase("", fontLineLCR));
		cell.setFixedHeight(5);
		pptCadre.addCell(cell);
		
		PdfPTable pptLigne2 = new PdfPTable(9);
		pptLigne2.setWidthPercentage(100);
		int[] widthsLigne2 = {18,2,13,2,13,2,32,2,16};
		pptLigne2.setWidths(widthsLigne2);		
		PdfPTable pptZoneLCR = new PdfPTable(1);
		pptZoneLCR.setWidthPercentage(100);
		cellLCR = new PdfPCell(pptZoneLCR.getDefaultCell());
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		cellLCR.setPhrase(new Phrase("MONTANT POUR CONTR\u00D4LE", fontLineLCR));
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT+PdfPCell.BOTTOM);
		cellLCR.setFixedHeight(20);
		pptZoneLCR.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne2.addCell(cellLigne);
		cellLigne = new PdfPCell(pptLigne2.getDefaultCell());
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		pptLigne2.addCell(cellLigne);
		pptZoneLCR = new PdfPTable(1);
		pptZoneLCR.setWidthPercentage(100);
		cellLCR = new PdfPCell(pptZoneLCR.getDefaultCell());
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);		
		cellLCR.setPhrase(new Phrase("DATE DE CR\u00C9ATION", fontLineLCR));
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT+PdfPCell.BOTTOM);
		cellLCR.setFixedHeight(20);
		pptZoneLCR.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne2.addCell(cellLigne);
		cellLigne = new PdfPCell(pptLigne2.getDefaultCell());
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		pptLigne2.addCell(cellLigne);
		pptZoneLCR = new PdfPTable(1);
		pptZoneLCR.setWidthPercentage(100);
		cellLCR = new PdfPCell(pptZoneLCR.getDefaultCell());
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);		
		cellLCR.setPhrase(new Phrase("\u00C9CH\u00C9ANCE", fontLineLCR));
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT+PdfPCell.BOTTOM);
		cellLCR.setBorderWidth(1.2f);
		cellLCR.setFixedHeight(20);
		pptZoneLCR.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne2.addCell(cellLigne);
		cellLigne = new PdfPCell(pptLigne2.getDefaultCell());
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		pptLigne2.addCell(cellLigne);		
		pptZoneLCR = new PdfPTable(1);
		pptZoneLCR.setWidthPercentage(100);		
		PdfPTable pptZoneLCR2 = new PdfPTable(3);
		pptZoneLCR2.setWidthPercentage(100);
		int[] widthsLigneLCR = {25,50,25};
		pptZoneLCR2.setWidths(widthsLigneLCR);		
		cellLCR = new PdfPCell(pptZoneLCR2.getDefaultCell());
		cellLCR.setBorder(PdfPCell.BOTTOM);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellLCR.setPhrase(new Phrase("L.C.R. seulement", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.BOTTOM);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setColspan(3);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cellLCR.setFixedHeight(5);
		pptZoneLCR2.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR2);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptZoneLCR.addCell(cellLigne);		
		pptZoneLCR2 = new PdfPTable(9);
		pptZoneLCR2.setWidthPercentage(100);
		int[] widthsLigneLCR2 = {5,30,5,8,8,8,8,8,20};
		pptZoneLCR2.setWidths(widthsLigneLCR2);		
		cellLCR = new PdfPCell(pptZoneLCR2.getDefaultCell());
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.BOTTOM);
		cellLCR.setFixedHeight(20);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellLCR.setVerticalAlignment(PdfPCell.ALIGN_BOTTOM);
		cellLCR.setPhrase(new Phrase("R\u00C9F. TIR\u00C9", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.RIGHT+PdfPCell.BOTTOM);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.BOTTOM+PdfPCell.RIGHT);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.BOTTOM+PdfPCell.RIGHT);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.BOTTOM+PdfPCell.RIGHT);
		pptZoneLCR2.addCell(cellLCR);		
		cellLigne = new PdfPCell(pptZoneLCR2);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptZoneLCR.addCell(cellLigne);		
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne2.addCell(cellLigne);
		cellLigne = new PdfPCell(pptLigne2.getDefaultCell());
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		pptLigne2.addCell(cellLigne);
		pptZoneLCR = new PdfPTable(1);
		pptZoneLCR.setWidthPercentage(100);
		cellLCR = new PdfPCell(pptZoneLCR.getDefaultCell());
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);		
		cellLCR.setPhrase(new Phrase("F. MONTANT", fontLineGrasLCR));
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT+PdfPCell.BOTTOM);
		cellLCR.setBorderWidth(1.2f);
		cellLCR.setFixedHeight(20);
		pptZoneLCR.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		pptLigne2.addCell(cellLigne);
		cell = new PdfPCell(pptLigne2);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);
		cell = new PdfPCell(pptCadre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setPhrase(new Phrase("", fontLineLCR));
		cell.setFixedHeight(5);
		pptCadre.addCell(cell);
		
		PdfPTable pptLigne3 = new PdfPTable(11);
		pptLigne3.setWidthPercentage(100);
		int[] widthsLigne3 = {2,40,2,2,2,30,2,2,2,14,2};
		pptLigne3.setWidths(widthsLigne3);
		cellLigne = new PdfPCell(pptLigne3.getDefaultCell());
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		cellLigne.setFixedHeight(20);
		cellLigne.setBorder(PdfPCell.LEFT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.RIGHT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.LEFT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.RIGHT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.LEFT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne3.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.RIGHT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptLigne3.addCell(cellLigne);		
		cell = new PdfPCell(pptLigne3);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);
		cell = new PdfPCell(pptCadre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setPhrase(new Phrase("", fontLineLCR));
		cell.setFixedHeight(5);
		pptCadre.addCell(cell);	
		
		PdfPTable pptLigne4 = new PdfPTable(5);
		pptLigne4.setWidthPercentage(100);
		int[] widthsLigne4 = {38,1,3,28,30};
		pptLigne4.setWidths(widthsLigne4);
		pptZoneLCR = new PdfPTable(1);
		pptZoneLCR.setWidthPercentage(100);
		pptZoneLCR2 = new PdfPTable(3);
		pptZoneLCR2.setWidthPercentage(100);
		int[] widthsLigneLCR3 = {35,30,35};
		pptZoneLCR2.setWidths(widthsLigneLCR3);
		cellLCR = new PdfPCell(pptZoneLCR2.getDefaultCell());
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		cellLCR.setBorder(PdfPCell.BOTTOM);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPhrase(new Phrase("R.I.B. du TIR\u00C9", fontLineLCR));
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.BOTTOM);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR2);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR.addCell(cellLigne);		
		pptZoneLCR2 = new PdfPTable(4);
		pptZoneLCR2.setWidthPercentage(100);
		int[] widthsLigneLCR4 = {20,20,50,10};
		pptZoneLCR2.setWidths(widthsLigneLCR4);
		cellLCR = new PdfPCell(pptZoneLCR2.getDefaultCell());		
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cellLCR.setFixedHeight(20);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		pptZoneLCR2.addCell(cellLCR);
		pptZoneLCR2.addCell(cellLCR);
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellLCR.setPhrase(new Phrase("code \u00E9tabl.", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("code guichet", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("N\u00B0 de compte", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("Cl\u00E9 R.I.B.", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLigne = new PdfPCell(pptZoneLCR2);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR.addCell(cellLigne);
		cellLigne = new PdfPCell(pptZoneLCR.getDefaultCell());
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setFixedHeight(5);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR.addCell(cellLigne);		
		pptZoneLCR2 = new PdfPTable(2);
		pptZoneLCR2.setWidthPercentage(100);
		int[] widthsLigneLCR5 = {50,50};
		pptZoneLCR2.setWidths(widthsLigneLCR5);
		cellLCR = new PdfPCell(pptZoneLCR2.getDefaultCell());		
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		cellLCR.setPhrase(new Phrase("Valeur en", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cellLCR.setPhrase(new Phrase("NOM", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("et ADRESSE", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("du TIRE", fontLineLCR));
		pptZoneLCR2.addCell(cellLCR);		
		cellLigne = new PdfPCell(pptZoneLCR2);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR.addCell(cellLigne);
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setFixedHeight(85);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne4.addCell(cellLigne);		
		cellLigne = new PdfPCell(pptLigne4.getDefaultCell());
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setPhrase(new Phrase("", fontLineLCR));
		cellLigne.setFixedHeight(85);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne4.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.TOP+PdfPCell.LEFT+PdfPCell.BOTTOM);
		cellLigne.setBorderWidth(1.2f);
		pptLigne4.addCell(cellLigne);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		pptLigne4.addCell(cellLigne);
		pptZoneLCR = new PdfPTable(3);
		pptZoneLCR.setWidthPercentage(100);
		int[] widthsLigneLCR6 = {25,50,25};
		pptZoneLCR.setWidths(widthsLigneLCR6);
		cellLCR = new PdfPCell(pptZoneLCR.getDefaultCell());
		cellLCR.setPaddingLeft(0);
		cellLCR.setPaddingRight(0);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setBorder(PdfPCell.BOTTOM);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellLCR.setPhrase(new Phrase("DOMICILIATION", fontLineLCR));
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		cellLCR.setBorder(PdfPCell.BOTTOM);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setColspan(3);
		cellLCR.setFixedHeight(20);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT+PdfPCell.TOP+PdfPCell.BOTTOM);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setColspan(1);
		cellLCR.setBorder(PdfPCell.NO_BORDER);
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("Droit de Timbre et Signature", fontLineLCR));
		pptZoneLCR.addCell(cellLCR);
		cellLCR.setPhrase(new Phrase("", fontLineLCR));
		pptZoneLCR.addCell(cellLCR);		
		cellLigne = new PdfPCell(pptZoneLCR);
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setBorder(PdfPCell.NO_BORDER);
		cellLigne.setFixedHeight(85);
		pptLigne4.addCell(cellLigne);		
		cell = new PdfPCell(pptLigne4);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);		
		cell = new PdfPCell(pptCadre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setPhrase(new Phrase("", fontLineLCR));
		cell.setFixedHeight(10);
		pptCadre.addCell(cell);		
		
		PdfPTable pptLigne5 = new PdfPTable(2);
		pptLigne5.setWidthPercentage(100);
		int[] widthsLigne5 = {40,60};
		pptLigne5.setWidths(widthsLigne5);
		cellLigne = new PdfPCell(pptLigne5.getDefaultCell());
		cellLigne.setPaddingLeft(0);
		cellLigne.setPaddingRight(0);
		cellLigne.setVerticalAlignment(PdfPCell.ALIGN_TOP);
		cellLigne.setBorder(PdfPCell.BOTTOM);		
		Chunk c3 = new Chunk("ACCEPTATION OU AVAL  ", fontLineLCR);
		Chunk c4;
		try {
			Image fleche = Image.getInstance(id.urlFlecheHautLCR);
			fleche.scaleAbsolute(5, 6);
			c4 = new Chunk(fleche, 0, 0);
		} catch (IOException e) {
			c4 = new Chunk("");
		}
		Phrase p2 = new Phrase();
		p2.add(c3); p2.add(c4);
		cellLigne.setPhrase(p2);
		pptLigne5.addCell(cellLigne);		
		Chunk c5 = new Chunk("ne rien inscrire au-dessous de cette ligne  ", fontLineLCR);
		Chunk c6;
		try {
			Image fleche = Image.getInstance(id.urlFlecheBasLCR);
			fleche.scaleAbsolute(5, 6);
			c6 = new Chunk(fleche, 0, 0);
		} catch (IOException e) {
			c6 = new Chunk("");
		}
		Phrase p3 = new Phrase();
		p3.add(c5); p3.add(c6);
		cellLigne.setPhrase(p3);
		pptLigne5.addCell(cellLigne);		
		cell = new PdfPCell(pptLigne5);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);
		
		return pptCadre;
	}
	
	
	private PdfPTable makeVirement() throws DocumentException {
		PdfPTable pptCadre = new PdfPTable(2);
		pptCadre.setWidthPercentage(100);
		int[] widths = {80,20};
		pptCadre.setWidths(widths);
		
		PdfPTable pptVirement = new PdfPTable(5);
		pptVirement.setWidthPercentage(100);
		int[] widths2 = {15,20,25,10,30};
		pptVirement.setWidths(widths2);
		PdfPCell cellVirement = new PdfPCell(pptVirement.getDefaultCell());
		cellVirement.setVerticalAlignment(PdfPCell.ALIGN_CENTER);
		cellVirement.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		cellVirement.setMinimumHeight(15);
		cellVirement.setPaddingLeft(0);
		cellVirement.setPaddingRight(0);		
		cellVirement.setPhrase(new Phrase("Nos coordonn\u00E9es bancaires", fontVirement));
		cellVirement.setColspan(5);
		cellVirement.setBorder(PdfPCell.NO_BORDER);
		pptVirement.addCell(cellVirement);
		
		cellVirement.setBorder(PdfPCell.BOX);
		cellVirement.setColspan(1);
		cellVirement.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cellVirement.setPhrase(new Phrase("Code Banque", fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase("Code Guichet", fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase("N\u00B0 de compte", fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase("Cl\u00E9", fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase("Domiciliation", fontVirement));
		pptVirement.addCell(cellVirement);
		
		cellVirement.setPhrase(new Phrase(id.codeBanque, fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase(id.codeGuichet, fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase(id.numCompte, fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase(id.cleRIB, fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase(id.domiciliation, fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase("IBAN", fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setColspan(2);
		cellVirement.setPhrase(new Phrase(id.IBAN, fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setColspan(1);
		cellVirement.setPhrase(new Phrase("BIC", fontVirement));
		pptVirement.addCell(cellVirement);
		cellVirement.setPhrase(new Phrase(id.BIC, fontVirement));
		pptVirement.addCell(cellVirement);
		
		PdfPCell cell = new PdfPCell(pptVirement);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		pptCadre.addCell(cell);
		
		cell = new PdfPCell(pptCadre.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPaddingLeft(0);
		cell.setPaddingRight(0);
		cell.setPhrase(new Phrase(""));
		pptCadre.addCell(cell);
		
		return pptCadre;
	}


	private PdfPTable makeInfosLegales() {

		PdfPTable pptLegales = new PdfPTable(1);
		pptLegales.setWidthPercentage(100);

		PdfPCell cell = new PdfPCell(pptLegales.getDefaultCell());		

		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setPhrase(new Phrase(id.raisonSociale +" - "+ id.typeSociete + (id.capital>0?" au capital de "+ numFormatCap.format(id.capital) +" Euros":"") +" - immatricul\u00E9e au RCS "+ id.rcs +" "+ id.siren +" - N\u00B0 TVA : "+ id.tvaIntra, fontLegale));

		pptLegales.addCell(cell);

		return pptLegales;	
	}


	// mthodes de paramtrage

	/*public void setMultiEx(boolean multi) {

		this.multi = multi;
	}*/


} // fin FactureStd
