/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Gnration d'un historique des articles fabriqus en pdf
 */

package org.opensi.facturation.actions.fabrication;

import java.sql.Date;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;

import org.opensi.util.pdf.MultiPagesDocument;
import org.opensi.util.pdf.PdfWidths;
import org.opensi.util.pdf.TableBuilder;

import com.lowagie.text.DocumentException;
import com.lowagie.text.Font;
import com.lowagie.text.Image;
import com.lowagie.text.Phrase;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;


public class PdfHistoriqueFabrication extends MultiPagesDocument {
	
	private static final Font font9N = new Font(Font.HELVETICA, 9, Font.NORMAL);
	private static final Font font9G = new Font(Font.HELVETICA, 9, Font.BOLD);
	private static final Font fontTitre = new Font(Font.HELVETICA, 16, Font.BOLD);
	
	private DataHistoriqueFabrication dhf;	   // les donnes de l'historique de fabrication
	
	private float[] widthsCols;
	private int nbCols;
	
	private DecimalFormat numFormat4 = new DecimalFormat("0.####");
	private SimpleDateFormat formatDate = new SimpleDateFormat("dd/MM/yyyy");
	
	
	public PdfHistoriqueFabrication(String tempdir, DataHistoriqueFabrication dhf) throws Exception {

		super(tempdir);
		
		this.dhf = dhf;		
		
		DecimalFormatSymbols dfs = new DecimalFormatSymbols();
		dfs.setDecimalSeparator('.');
		dfs.setGroupingSeparator(' ');

		numFormat4.setDecimalFormatSymbols(dfs);
		numFormat4.setNegativeSuffix("");
		numFormat4.setNegativePrefix("-");
		numFormat4.setPositiveSuffix(" ");
		numFormat4.setGroupingUsed(true);
		numFormat4.setGroupingSize(3);
		
		PdfWidths pw = new PdfWidths();
		
		pw.addColonneFixe(11);
		pw.addColonneVariableMin(1,20);
		pw.addColonneVariableMin(1,20);
		pw.addColonneFixe(11);
		pw.addColonneFixe(15);
		
		nbCols = pw.getNbCols();
		widthsCols = pw.getWidthsCols();
	}


	
	/* implmentation des fonctions de MultiPagesDocument */
	
	public PdfPTable firstHeader() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeEntete());
		tb.add(makeSpace(10));
		tb.add(makeEntetesColonnes());
		
		return tb.getTable();	
	}
	
	
	public PdfPTable middleHeader(int lastLine) throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(firstHeader());
		
		return tb.getTable();
	}
	
	
	public PdfPTable lastFooter() throws DocumentException {
		
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeFinPage());
		
		return tb.getTable();
	}
	
	
	public PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException {
			
		TableBuilder tb = new TableBuilder();
		
		tb.add(makeFinPage());
		
		return tb.getTable();		
	}
	
	
	public PdfPTable bodyLine(int numLine) throws DocumentException {
	
		LigneHistoriqueFabrication lh = dhf.lignes.get(numLine);
		
		return makeLine(lh);
	}
	
	
	public PdfPTable emptyLine(float emptyHeight) throws DocumentException {

		PdfPTable pptSpace = new PdfPTable(nbCols);
		pptSpace.setWidthPercentage(100);
		pptSpace.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(emptyHeight);		
		cell.setPhrase(new Phrase(""));

		for (int i=0; i<nbCols; i++) {
			pptSpace.addCell(cell);
		}

		return pptSpace;
	}
	
	
	public int nbLines() {
		
		return dhf.getNbLignes();
	}
	
	
	public int nbEx() {
	
		return 1;
	}
	
	
	public Image imageFond() {
	
		return null;
	}
	
	
	/* mthodes internes */
	

	private PdfPTable makeSpace(int space) {
	
		PdfPTable pptSpace = new PdfPTable(1);
		pptSpace.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptSpace.getDefaultCell());
		cell.setBorder(PdfPCell.NO_BORDER);
		cell.setFixedHeight(space);
		cell.setPhrase(new Phrase(""));
	
		pptSpace.addCell(cell);
	
		return pptSpace;
	}
	
	
	private PdfPTable makeEntete() throws DocumentException {
		
		PdfPTable bandeau = new PdfPTable(1);

		bandeau.setWidthPercentage(100);

		PdfPTable entete = new PdfPTable(3);

		PdfPCell cellEntete = new PdfPCell(entete.getDefaultCell());

		cellEntete.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cellEntete.setBorder(PdfPCell.NO_BORDER);
		cellEntete.setMinimumHeight(10);

		cellEntete.setPhrase(new Phrase("Dossier : "+ dhf.dossierId, font9N));
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(dhf.raisonSociale, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);

		cellEntete.setPhrase(new Phrase("Date : " + dhf.dateCourante, font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(dhf.adresse1, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);

		cellEntete.setPhrase(new Phrase("", font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(dhf.adresse2, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);
		cellEntete.setColspan(1);

		cellEntete.setPhrase(new Phrase("Page : "+ currentPage +" / "+ totalPages, font9N));
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
		entete.addCell(cellEntete);
		
		if (!dhf.adresse3.equals("")) {
			cellEntete.setPhrase(new Phrase(dhf.adresse3, font9N));
			cellEntete.setColspan(2);
			cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
			entete.addCell(cellEntete);
			cellEntete.setColspan(1);
			
			cellEntete.setPhrase(new Phrase("", font9N));
			cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_LEFT);
			entete.addCell(cellEntete);
		}

		cellEntete.setPhrase(new Phrase(dhf.codePostal +" "+ dhf.ville, font9N));
		cellEntete.setColspan(2);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase("Historique de fabrication des articles nomenclatur\u00E9s", fontTitre));
		cellEntete.setColspan(3);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase(""));
		cellEntete.setFixedHeight(10);
		entete.addCell(cellEntete);

		cellEntete.setPhrase(new Phrase("P\u00E9riode du "+ dhf.debutPeriodeStr +" au "+ dhf.finPeriodeStr, font9N));
		cellEntete.setMinimumHeight(0);
		cellEntete.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		entete.addCell(cellEntete);

		PdfPCell cellBandeau = new PdfPCell(entete);
		cellBandeau.setPadding(10);

		bandeau.addCell(cellBandeau);
		
		return bandeau;		
	}
	
	
	private PdfPTable makeEntetesColonnes() throws DocumentException {

		PdfPTable pptCols = new PdfPTable(nbCols);
		pptCols.setWidthPercentage(100);
		pptCols.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptCols.getDefaultCell());
		cell.setMinimumHeight(30);
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);

		cell.setPhrase(new Phrase("Date Fabrication", font9G));
		pptCols.addCell(cell);
		
		cell.setPhrase(new Phrase("R\u00E9f. Article", font9G));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("D\u00E9signation", font9G));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("Quantit\u00E9", font9G));
		pptCols.addCell(cell);

		cell.setPhrase(new Phrase("Type", font9G));
		pptCols.addCell(cell);

		return pptCols;
	}
	
	
	private PdfPTable makeFinPage() {
		
		PdfPTable pptFP = new PdfPTable(1);
		pptFP.setWidthPercentage(100);
		
		PdfPCell cell = new PdfPCell(pptFP.getDefaultCell());
		cell.setBorder(PdfPCell.TOP);
		cell.setPhrase(new Phrase(""));
		cell.setPaddingBottom(0);
		cell.setPaddingTop(0);
		cell.setFixedHeight(1);
				
		pptFP.addCell(cell);
	
		return pptFP;
	}
	
	
	private PdfPTable makeLine(LigneHistoriqueFabrication lhf) throws DocumentException {

		PdfPTable pptLine = new PdfPTable(nbCols);
		pptLine.setWidthPercentage(100);
		pptLine.setWidths(widthsCols);

		PdfPCell cell = new PdfPCell(pptLine.getDefaultCell());
		cell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
		cell.setBorder(PdfPCell.LEFT+PdfPCell.RIGHT);

		cell.setPhrase(new Phrase(formatDate.format(new Date(lhf.dateFabrication)), font9N));
		pptLine.addCell(cell);

		cell.setPhrase(new Phrase(lhf.refArticle, font9N));
		pptLine.addCell(cell);
		
		cell.setPhrase(new Phrase(lhf.designation, font9N));
		pptLine.addCell(cell);

		cell.setHorizontalAlignment(PdfPCell.ALIGN_RIGHT);
		cell.setPaddingRight(4);

		cell.setPhrase(new Phrase(numFormat4.format(lhf.quantite), font9N));
		pptLine.addCell(cell);
		
		cell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
		cell.setPaddingRight(0);

		cell.setPhrase(new Phrase(lhf.type, font9N));
		pptLine.addCell(cell);

		return pptLine;
	}


} // fin PdfHistoriqueFabrication
