/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

/**
 * Document interne  un GlobalDocument
 */

package org.opensi.util.pdf;


import java.util.ArrayList;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Image;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfWriter;


public abstract class MultiPagesDocument implements InternalDocument {


	protected int totalPages = 1;  // nombre total de pages
	protected int currentPage = 1; // numro de la page courante

	private String tempdir; // chemin du rpertoire temporaire

	private Image fond;  // image de fond

	private PdfWriter writer;

	private ArrayList<ArrayList<PdfPTable>> pages; // liste des pages du document contenant les lignes de la page (ArrayList)

	private ArrayList<Float> emptyLines; // liste des hauteurs de lignes vides par page


	public MultiPagesDocument(String tempdir) {

		this.tempdir = tempdir;
	}


	public final void buildDocument(Document document, PdfWriter writer) throws DocumentException {


		this.pages = new ArrayList<ArrayList<PdfPTable>>(2);
		this.emptyLines = new ArrayList<Float>(2);
		this.fond = imageFond();
		this.writer = writer;


		GlobalDocument gd = new GlobalDocument(tempdir, "temp_"+ System.currentTimeMillis());

		Document temp = gd.getDocument();

		PdfPTable fh = firstHeader();

		temp.add(fh);

		// hauteur du premier entte
		float firstHeaderHeight = fh.getTotalHeight();

		PdfPTable lf = lastFooter();

		temp.add(lf);

		// hauteur du dernier pied de page
		float lastFooterHeight = lf.getTotalHeight();

		// hauteur disponible d'une page
		float availablePageHeight = document.getPageSize().getHeight() - document.topMargin() - document.bottomMargin();

		// nombre de lignes du corps du document
		int nbLines = nbLines();


		float availableHeight = availablePageHeight - firstHeaderHeight;

		ArrayList<PdfPTable> lines = new ArrayList<PdfPTable>(10); // lignes de la page courante

		int nbLinesPage = 0; // nombre de lignes dans la page;

		int i = 0;

		while (i<nbLines) {
			// il reste des lignes  poser

			PdfPTable currentLine = bodyLine(i);
			temp.add(currentLine);
			float currentLineHeight = currentLine.getTotalHeight();

			PdfPTable mf = middleFooter(nbLinesPage+1, i); //  calculer avec la ligne courante
			temp.add(mf);
			float middleFooterHeight = mf.getTotalHeight();

			if ((middleFooterHeight + currentLineHeight) > availableHeight) {
				// ya pas la place de mettre la ligne courante sur la page en cours
				// on termine la page et on passe  la suivante

				// on recalcule la hauteur du middleFooter
				mf = middleFooter(nbLinesPage, i-1); //  calculer sans la ligne courante
				temp.add(mf);
				middleFooterHeight = mf.getTotalHeight();

				availableHeight -= middleFooterHeight;
				emptyLines.add(new Float(availableHeight));
				pages.add(lines);
				nbLinesPage = 0;

				// on commence la nouvelle page avec un middleHeader

				PdfPTable mh = middleHeader(i-1);
				temp.add(mh);
				float middleHeaderHeight = mh.getTotalHeight();

				availableHeight = availablePageHeight - middleHeaderHeight;

				lines = new ArrayList<PdfPTable>(10);
			}

			lines.add(currentLine);
			nbLinesPage++;
			availableHeight -= currentLineHeight;
			i++;
		}


		// toutes les lignes ont t poses
		// on essaie de poser le lastFooter (taille connue) sinon on cre un middleFooter et une nouvelle page

		if (lastFooterHeight > availableHeight) {

			PdfPTable mf = middleFooter(nbLinesPage, nbLines-1); //  calculer avec toutes les lignes
			temp.add(mf);
			float middleFooterHeight = mf.getTotalHeight();

			availableHeight -= middleFooterHeight;
			emptyLines.add(new Float(availableHeight));
			pages.add(lines);

			// on commence la nouvelle page avec un middleHeader

			PdfPTable mh = middleHeader(nbLines-1);
			temp.add(mh);
			float middleHeaderHeight = mh.getTotalHeight();

			availableHeight = availablePageHeight - middleHeaderHeight;

			lines = new ArrayList<PdfPTable>(1);  // plus de lignes
		}

		availableHeight -= lastFooterHeight;
		emptyLines.add(new Float(availableHeight));
		pages.add(lines);


		gd.close(GlobalDocument.DELETE);


		// Construction relle du document

		totalPages = pages.size();
		int lastPage = totalPages-1;


		for (int nx=0; nx<nbEx(); nx++) {
			i = -1;
			currentPage = 1;

			for (int np=0; np<totalPages; np++) {

				this.fond = imageFond();
				addFond();

				if (np>0)
					document.add(middleHeader(i));
				else
					document.add(firstHeader());

				lines = pages.get(np);

				for (int nl=0; nl<lines.size(); nl++) {
					document.add(lines.get(nl));
				}

				i += lines.size();

				document.add(emptyLine((emptyLines.get(np)).floatValue()));

				if (np<lastPage)
					document.add(middleFooter(lines.size(), i));
				else
					document.add(lastFooter());

				currentPage++;
			}
		}

	}


	private void addFond() throws DocumentException {
		if (fond!=null)	writer.getDirectContentUnder().addImage(fond);
	}

  public abstract PdfPTable firstHeader() throws DocumentException;
	public abstract PdfPTable middleHeader(int lastLine) throws DocumentException;
	public abstract PdfPTable lastFooter() throws DocumentException;
	public abstract PdfPTable middleFooter(int nbLinesPage, int lastLine) throws DocumentException;
	public abstract PdfPTable bodyLine(int numLine) throws DocumentException;
	public abstract PdfPTable emptyLine(float emptyHeight) throws DocumentException;
	public abstract int nbLines();
	public abstract int nbEx();
	public abstract Image imageFond();


} // fin MultiPagesDocument
